package com.zjty.fp.acq.vomp.subject.service.updater.impl;

import com.google.common.collect.Lists;
import com.zjty.fp.acq.misc.utils.Downloader;
import com.zjty.fp.acq.misc.utils.FileReader;
import com.zjty.fp.acq.misc.utils.JacksonUtil;
import com.zjty.fp.acq.vomp.base.enums.VompCatalogs;
import com.zjty.fp.acq.vomp.base.enums.VompPlatform;
import com.zjty.fp.acq.vomp.subject.entity.localdo.*;
import com.zjty.fp.acq.vomp.subject.entity.remotedo.*;
import com.zjty.fp.acq.vomp.subject.repository.local.*;
import com.zjty.fp.acq.vomp.subject.service.api.VompTransHelper;
import com.zjty.fp.acq.vomp.subject.service.cache.VompTransCache;
import com.zjty.fp.acq.vomp.subject.service.updater.VompUpdater;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;


/**
 * <p>Description : fp-fp
 * <p>Date : 2018/11/20 10:18
 * <p>@author : C
 * Dl意为Download。该组件的同步数据模式为从FTP服务器下载文件然后解析保存。
 */
@SuppressWarnings("SpringAutowiredFieldsWarningInspection")
@Service
@Slf4j
public class VompDataDlUpdater implements VompUpdater {
    @Autowired
    VompTransCache transCache;
    @Autowired
    VompTransHelper transHelper;
    @Autowired
    Downloader downloader;
    @Autowired
    FileReader fileReader;
    @Autowired
    VompAlarmRepository alarmRepository;
    @Autowired
    VompHandingInfoRepository handingInfoRepository;
    @Autowired
    VompPcInfoRepository pcInfoRepository;
    @Autowired
    VompStatusRepository statusRepository;
    @Autowired
    VompZoneRepository zoneRepository;

    @Override
    public void updateData() {
        //download file
        downloadFileFromFtpServer();
        log.info("[vomp] 文件下载完毕");
        //read file & parse & save to cache
        readAllFilesAndSaveCache();
        //trans date to local format & save
        updateAllData();
        log.info("[vomp] 数据更新完毕");
    }

    /*
    <Private Methods Area>
    */

    /**
     * Remote Data
     */
    private void downloadFileFromFtpServer() {
        downloader.download(VompPlatform.PLATFORM.getName(), VompCatalogs.CHKROUTES.getName());
        downloader.download(VompPlatform.PLATFORM.getName(), VompCatalogs.CHKSTATES.getName());
        downloader.download(VompPlatform.PLATFORM.getName(), VompCatalogs.PCLOG.getName());
        downloader.download(VompPlatform.PLATFORM.getName(), VompCatalogs.SERVERSTATUS.getName());
        downloader.download(VompPlatform.PLATFORM.getName(), VompCatalogs.ZONE.getName());
    }

    private void readAllFilesAndSaveCache() {
        readChkRoutesAndSaveCache();
        readChkStatesAndSaveCache();
        readPcLogAndSaveCache();
        readServerStatusAndSaveCache();
        readZoneAndSaveCache();
    }

    private String readFileFirstLine(String catalog) {
        Optional<String> firstLine = fileReader.readFileData(VompPlatform.PLATFORM.getName(), catalog).stream().findFirst();
        return firstLine.orElse("");
    }

    private void readChkRoutesAndSaveCache() {
        String text = readFileFirstLine(VompCatalogs.CHKROUTES.getName());
        ChkRoutes[] chkRoutesArrays = JacksonUtil.readValue(text, ChkRoutes[].class);
        List<ChkRoutes> chkRoutesList = Lists.newArrayList();
        if (chkRoutesArrays != null) {
            chkRoutesList = Arrays.asList(chkRoutesArrays);
        }
        transCache.setChkRoutesList(chkRoutesList);
    }

    private void readChkStatesAndSaveCache() {
        String text = readFileFirstLine(VompCatalogs.CHKSTATES.getName());
        ChkStates[] chkStatesArrays = JacksonUtil.readValue(text, ChkStates[].class);
        List<ChkStates> chkStatesList = Lists.newArrayList();
        if (chkStatesArrays != null) {
            chkStatesList = Arrays.asList(chkStatesArrays);
        }
        transCache.setChkStatesList(chkStatesList);
    }

    private void readPcLogAndSaveCache() {
        String text = readFileFirstLine(VompCatalogs.PCLOG.getName());
        PcLog[] pcLogArrays = JacksonUtil.readValue(text, PcLog[].class);
        List<PcLog> pcLogList = Lists.newArrayList();
        if (pcLogArrays != null) {
            pcLogList = Arrays.asList(pcLogArrays);
        }
        transCache.setPcLogList(pcLogList);
    }

    private void readServerStatusAndSaveCache() {
        String text = readFileFirstLine(VompCatalogs.SERVERSTATUS.getName());
        ServerStatus[] serverStatusArrays = JacksonUtil.readValue(text, ServerStatus[].class);
        List<ServerStatus> serverStatusList = Lists.newArrayList();
        if (serverStatusArrays != null) {
            serverStatusList = Arrays.asList(serverStatusArrays);
        }
        transCache.setServerStatusList(serverStatusList);
    }

    private void readZoneAndSaveCache() {
        String text = readFileFirstLine(VompCatalogs.ZONE.getName());
        Zone[] zoneArrays = JacksonUtil.readValue(text, Zone[].class);
        List<Zone> zoneList = Lists.newArrayList();
        if (zoneArrays != null) {
            zoneList = Arrays.asList(zoneArrays);
        }
        transCache.setZoneList(zoneList);
    }

    /**
     * Local Data
     */
    private void updateAllData() {
        updateVompAlarmDl();
        updateVompHandingInfoDl();
        updateVompPcInfoDl();
        updateVompStatusDl();
        updateZoneDl();
    }

    private void updateVompAlarmDl() {
        List<VompAlarm> alarmList4Save = getAlarmList();
        alarmRepository.save(alarmList4Save);
        log.info("[vomp] VompAlarm更新了{}条数据", alarmList4Save.size());
    }

    private void updateVompHandingInfoDl() {
        List<VompHandingInfo> handingInfoList4Save = getHandingInfoList();
        handingInfoRepository.save(handingInfoList4Save);
        log.info("[vomp] VompHandingInfo更新了{}条数据", handingInfoList4Save.size());
    }

    private void updateVompPcInfoDl() {
        List<VompPcInfo> pcInfoList4Save = getPcInfoList();
        pcInfoRepository.save(pcInfoList4Save);
        log.info("[vomp] VompPcInfo更新了{}条数据", pcInfoList4Save.size());
    }

    private void updateVompStatusDl() {
        List<VompStatus> statusList4Save = getStatusList();
        statusRepository.save(statusList4Save);
        log.info("[vomp] VompStatus更新了{}条数据", statusList4Save.size());
    }

    private void updateZoneDl() {
        List<VompZone> zoneList4Save = getZoneList();
        zoneRepository.save(zoneList4Save);
        log.info("[vomp] VompZone更新了{}条数据", zoneList4Save.size());
    }

    /**
     * VompAlarm
     */
    private List<VompAlarm> getAlarmList() {
        List<VompAlarm> vompAlarmList = Lists.newArrayList();
        transCache.getChkStates_PcLog_Map().forEach((chkstates, pcLogs) -> pcLogs.forEach(pclog -> vompAlarmList.add(transHelper.transToVompAlarm(chkstates, pclog))));
        return vompAlarmList;
    }

    /**
     * VompHandingInfo
     */
    private List<VompHandingInfo> getHandingInfoList() {
        List<VompHandingInfo> vompHandingInfoList = Lists.newArrayList();
        transCache.getChkStates_ChkRoutes_Map().forEach((chkstates, chkRoutes) -> chkRoutes.forEach(chkRoute -> vompHandingInfoList.add(transHelper.transToHandingInfo(chkstates, chkRoute))));
        return vompHandingInfoList;
    }

    /**
     * VompPcInfo
     */
    private List<VompPcInfo> getPcInfoList() {
        List<VompPcInfo> vompPcInfoList = Lists.newArrayList();
        transCache.getZone_PcLog_Map().forEach((zone, pcLogList) -> vompPcInfoList.add(transHelper.transToVompPcInfoVo(zone, pcLogList)));
        return vompPcInfoList;
    }

    /**
     * Status
     */
    private List<VompStatus> getStatusList() {
        return transCache.getServerStatusList().stream()
                .map(transHelper::transToStatus)
                .collect(Collectors.toList());
    }

    /**
     * VompZone
     */
    private List<VompZone> getZoneList() {
        return transCache.getZoneList().stream()
                .map(transHelper::transToVompZone)
                .map(transHelper::addParentZoneName)
                .map(transHelper::addGrandParentZoneName)
                .collect(Collectors.toList());
    }

}
