package com.zjty.fp.api.pssp.subject.service.impl;

import com.zjty.fp.api.pssp.base.constant.DictConst;
import com.zjty.fp.api.pssp.subject.entity.location.Region;
import com.zjty.fp.api.pssp.subject.entity.remote.RemoteRegion;
import com.zjty.fp.api.pssp.subject.repository.location.RegionRepository;
import com.zjty.fp.api.pssp.subject.repository.remote.RemoteRegionRepository;
import com.zjty.fp.api.pssp.subject.service.RegionService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.Objects.nonNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * RegionServiceImpl. 区域接口实现类
 *
 * @author Matrix <xhyrzldf@gmail.com>
 * @since 2019-04-08 at 16:33
 */
@Slf4j
@Service
@Transactional(rollbackOn = Exception.class)
public class RegionServiceImpl implements RegionService {

    @Autowired
    private RegionRepository regionRepository;

    @Autowired
    private RemoteRegionRepository remoteRegionRepository;


    @Override
    public void fetchAllData() {
        log.info("[pssp]准备抓取全部源地区数据");

        List<Region> regionList = remoteRegionRepository.findAll()
                .stream()
                .map(RemoteRegion::toDo)
                .collect(toList());

        log.info("[pssp]源地区数据抓取完成,size = {},准备写入融合平台数据库", regionList.size());

        // 异步
        CompletableFuture.runAsync(() -> {
            //更新本地DB
            log.info("[pssp]正在异步写入地区数据到本地数据库");
            regionRepository.save(regionList);
            log.info("[pssp]异步地区数据写入完成");

            //更新字典map
            log.info("[pssp] [定时任务] 开始更新字典Map任务,更新前的字典Map大小为,regionMap : {}", DictConst.REGION_MAP.size());
            DictConst.REGION_MAP = findDictMap();
            log.info("[pssp] [定时任务] 完成更新字典Map任务,更新后的字典Map大小为,regionMap : {}", DictConst.REGION_MAP.size());
        });
    }

    @Override
    public void fetchUpdatedData() {
        log.info("[pssp]正在执行地区数据的更新");


        //批量对比 首先获取远端DB所有区域，对比数量，数量一致任务结束,数量不一致获取自己没有的进行采集
        List<RemoteRegion> remoteRegionList = remoteRegionRepository.findAll();
        List<Region> localRegionList = regionRepository.findAll();

        int remoteSize = remoteRegionList.size();
        int localSize = localRegionList.size();
        if (remoteSize != localSize) {
            log.info("[pssp] 本次地区更新采集没有要更新的数据,本地区域size : {},远端区域size : {}", localSize, remoteSize);
        } else {
            CompletableFuture.runAsync(() -> {
                //映射为Map , key : code ,value : region object ,如果远程的区域对象code 在此get不到则，证明需要插入
                Map<String, Region> localRegionMap = localRegionList.stream().collect(toMap(Region::getCode, Function.identity()));

                List<Region> regionList = remoteRegionList.stream()
                        .filter(remoteRegion -> nonNull(localRegionMap.get(remoteRegion.getCode())))
                        .map(RemoteRegion::toDo)
                        .collect(toList());

                log.info("[pssp] 发现需要同步更新的地区对象,对象内容为 : {}", regionList);

                log.info("[pssp]正在异步写入地区更新数据到本地数据库");
                regionRepository.save(regionList);
                log.info("[pssp]异步地区更新数据写入完成");

                //更新字典map
                log.info("[pssp] [定时任务] 开始更新字典Map任务,更新前的字典Map大小为,regionMap : {}", DictConst.REGION_MAP.size());
                DictConst.REGION_MAP = findDictMap();
                log.info("[pssp] [定时任务] 完成更新字典Map任务,更新后的字典Map大小为,regionMap : {}", DictConst.REGION_MAP.size());

            });
        }
    }

    @Override
    public List<Region> findAllData() {
        return regionRepository.findAll();
    }

    @Override
    public Map<String, Region> findDictMap() {

        return regionRepository.findAll()
                .stream()
                .collect(Collectors
                        .toMap(Region::getCode, Function.identity()));
    }
}
