package com.zjty.fp.api.misc.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.net.ftp.FTPClient;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * <p>Description : Downloader
 * <p>Date : 2018/9/11 16:15
 * <p>@author : C
 * 访问FTP服务器并下载文件的工具类
 * 已作为bean在spring中注册
 * 需要使用时自动注入（@Autowired）即可
 * 需要的参数从properties文件中读取
 * 请保证properties文件中有配置这些参数
 * 有以下参数：
 * ftp.server.ip FTP服务器的ip地址
 * ftp.server.port FTP服务器的端口号
 * ftp.server.username 用户名
 * ftp.server.password 密码
 */
@SuppressWarnings({"unused", "UnusedReturnValue"})
@Component
@Scope("prototype")
@Slf4j
public class Downloader {
    @Value("${ftp.server.ip}")
    private String ip;
    @Value("${ftp.server.port}")
    private Integer port;
    @Value("${ftp.server.username}")
    private String userName;
    @Value("${ftp.server.password}")
    private String password;

    private FTPClient ftpClient = new FTPClient();

    /**
     * @param subName 平台文件夹名 （FTP服务器上存有多个平台的文件）
     * @param catalog 种类文件夹名 (每个平台又有多种种类的文件)
     * @return boolean 是否下载成功
     * 从指定的文件夹路径下载当天的文件
     * 文件夹路径与上传时的相同:平台文件夹名/种类文件夹名/
     * 文件名也为相同的自动生成 格式:平台名-种类-时间（yyyyMMdd）.txt
     * 默认下载目的地路径是:项目根目录/file/平台文件夹名/种类文件夹名/
     * 若项目是以jar包形式运行的状态，则是:项目jar包同级目录/file/平台文件夹名/种类文件夹名/
     */
    public boolean download(String subName, String catalog) {
        return downloadFile(subName, catalog);
    }

    /**
     * @param subName 平台文件夹名 （FTP服务器上存有多个平台的文件）
     * @param catalog 种类文件夹名 (每个平台又有多种种类的文件)
     * @return boolean 是否下载成功
     * 从指定的文件夹路径下载当天的文件
     * 文件夹路径与上传时的相同:平台文件夹名/种类文件夹名/
     * 文件名也为相同的自动生成 格式:平台名-种类-时间（yyyyMMdd）.txt
     * 默认下载目的地路径是:项目根目录/file/平台文件夹名/种类文件夹名/
     * 若项目是以jar包形式运行的状态，则是:项目jar包同级目录/file/平台文件夹名/种类文件夹名/
     */
    public boolean download(String subName, String catalog, String Time) {
        return downloadFile(subName, catalog, Time);
    }

    /*
私有方法
 */
    private boolean downloadFile(String subName, String catalog, String Time) {
        boolean downloaded = false;
        FileOutputStream fos = null;
        if (connectServer()) {
            try {
                ftpClient.setControlEncoding("UTF-8");
                ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
                String fileName = getFileNameByDate(subName, catalog, Time);
                boolean subEnter = ftpClient.changeWorkingDirectory(subName);
                if (subEnter) {
                    //log.info("进入平台级文件夹-当前工作路径为：" + ftpClient.printWorkingDirectory());
                    boolean catalogEnter = ftpClient.changeWorkingDirectory(catalog);
                    if (catalogEnter) {
                        //   log.info("进入类型级文件夹-当前工作路径为：" + ftpClient.printWorkingDirectory());
                        String localFilePath = "file/" + subName + "/" + catalog + "/";
                        //      log.info("确认本地路径");
                        createFilePath(new File(localFilePath));
                        fos = new FileOutputStream(localFilePath + fileName);
//                        log.info("开始下载文件到本地：" + fileName);
                        downloaded = ftpClient.retrieveFile(getFileNameByDate(subName, catalog, Time), fos);
                    }
                }
            } catch (IOException e) {
                log.info("下载文件异常:{}", e);
            } finally {
                try {
                    if (fos != null) {
                        fos.close();
                    }
                    ftpClient.disconnect();
                } catch (IOException e) {
                    log.info("关闭连接异常:{}", e);
                }
            }
        }
        return downloaded;
    }


    /*
    私有方法
     */
    private boolean downloadFile(String subName, String catalog) {
        boolean downloaded = false;
        FileOutputStream fos = null;
        if (connectServer()) {
            try {
                ftpClient.setControlEncoding("UTF-8");
                ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
                String fileName = getFileNameByDate(subName, catalog);
                boolean subEnter = ftpClient.changeWorkingDirectory(subName);
                if (subEnter) {
//                    log.info("进入平台级文件夹-当前工作路径为：" + ftpClient.printWorkingDirectory());
                    boolean catalogEnter = ftpClient.changeWorkingDirectory(catalog);
                    if (catalogEnter) {
//                        log.info("进入类型级文件夹-当前工作路径为：" + ftpClient.printWorkingDirectory());
                        String localFilePath = "file/" + subName + "/" + catalog + "/";
//                        log.info("确认本地路径");
                        createFilePath(new File(localFilePath));
                        fos = new FileOutputStream(localFilePath + fileName);
//                        log.info("下载文件到本地：" + fileName);
                        downloaded = ftpClient.retrieveFile(getFileNameByDate(subName, catalog), fos);
                    }
                }
                if (!downloaded) {
                    log.info("指定的文件不存在:{}", fileName);
                }
            } catch (IOException e) {
                log.info("下载文件异常:{}", e);
            } finally {
                try {
                    if (fos != null) {
                        fos.close();
                    }
                    ftpClient.disconnect();
                } catch (IOException e) {
                    log.info("关闭连接异常:{}", e);
                }
            }
        } else {
            connectServer();
        }
        return downloaded;
    }

    private boolean connectServer() {
        boolean isSuccess = false;
        try {
            ftpClient.connect(ip, port);
            isSuccess = ftpClient.login(userName, password);
        } catch (IOException e) {
            log.error("连接FTP服务器异常:{}", e);
        }
        return isSuccess;
    }

    private boolean checkConnect() {
        boolean connected = ftpClient.isConnected();
        if (!connected) {
            connectServer();
        }
        return ftpClient.isConnected();
    }

    /**
     * 根据日期生成文件名
     * 以天为单位
     * 文件名格式：平台名-种类-时间（yyyyMMdd）.txt
     */
    private String getFileNameByDate(String subName, String catalog) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        return subName + "-" + catalog + "-" + sdf.format(new Date()) + ".txt";
    }

    /**
     * 根据日期生成文件名
     * 以天为单位
     * 文件名格式：平台名-种类-时间（yyyyMMdd）.txt
     */
    private String getFileNameByDate(String subName, String catalog, String Time) {

        return subName + "-" + catalog + "-" + Time + ".txt";
    }

    /**
     * 生成指定的文件夹路径
     * 若文件夹不存在则创建
     */
    private boolean createFilePath(File file) {
        return file.exists() || file.mkdirs();
    }
}
