package com.zjty.fp.api.pssp.subject.controller;

import com.google.common.collect.ImmutableMap;
import com.zjty.fp.api.misc.utils.DateTimeUtil;
import com.zjty.fp.api.pssp.subject.entity.location.Alert;
import com.zjty.fp.api.pssp.subject.entity.vo.PsspPageVo;
import com.zjty.fp.api.pssp.subject.service.AlertService;
import com.zjty.fp.api.pssp.subject.service.WebsiteService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

import static org.springframework.http.ResponseEntity.ok;

/**
 * 第三方api接口
 *
 * @author : Matrix [xhyrzldf@foxmail.com]
 * @since : 18-8-22
 */
@RequestMapping("/api/pssp")
@RestController
public class PsspApiController {

    @Autowired
    private AlertService alertService;

    @Autowired
    private WebsiteService websiteService;

    /**
     * 获得所有网站数据
     */
    @GetMapping("/websites")
    public ResponseEntity findAll() {
        return ResponseEntity.ok(websiteService.findAllData());
    }

    @GetMapping("/alerts")
    public ResponseEntity<PsspPageVo> findAllData(
            @RequestParam(value = "startTime") String startTime,
            @RequestParam(value = "endTime") String endTime,
            @RequestParam(value = "page", defaultValue = "1") int page,
            @RequestParam(value = "size", defaultValue = "10") int size,
            @RequestParam(value = "sortDirection", defaultValue = "ASC") String sortDirection

    ) {

        page--;
        //默认为ASC
        Sort.Direction direction = Sort.Direction.ASC;
        if ("DESC".equals(sortDirection)) {
            direction = Sort.Direction.DESC;
        }

        LocalDate localStartTime = LocalDate.parse(startTime, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        LocalDate localEndTime = LocalDate.parse(endTime, DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        Page<Alert> alertPage = alertService.findAllDataBetweenTime(localStartTime, localEndTime, new PageRequest(page, size, direction, "tmFetch"));
        return ResponseEntity.ok(PsspPageVo.trans2PageVo(alertPage, page));
    }

    @GetMapping("/alerts/after")
    public ResponseEntity<PsspPageVo> findDataAfterId(
            @RequestParam(value = "primaryId") Long id,
            @RequestParam(value = "page", defaultValue = "1") int page,
            @RequestParam(value = "size", defaultValue = "10") int size) {

        page--;
        Page<Alert> alertPage = alertService.findAllAfterId(id, new PageRequest(page, size));
        return ResponseEntity.ok(PsspPageVo.trans2PageVo(alertPage, page));
    }

    @GetMapping("/alerts/recommend")
    public ResponseEntity findRecommendTime() {
        String recommendTime = DateTimeUtil.formatDateTimetoString(alertService.findEarliestTime(), "yyyy-MM-dd HH:mm:ss");
        return ok(ImmutableMap.of("recommendTime", recommendTime));
    }
}
