package com.zjty.fp.api.pssp.subject.service.impl;

import com.zjty.fp.api.misc.utils.FileReader;
import com.zjty.fp.api.misc.utils.JacksonUtil;
import com.zjty.fp.api.pssp.subject.entity.location.Alert;
import com.zjty.fp.api.pssp.subject.repository.location.AlertRepository;
import com.zjty.fp.api.pssp.subject.service.AlertService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.transaction.Transactional;
import java.security.InvalidParameterException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.zjty.fp.api.misc.entity.PsspCount.COUNT_ADDRESS_ALERT;
import static com.zjty.fp.api.misc.utils.DateTimeUtil.LocalToDate;
import static com.zjty.fp.api.misc.utils.DateTimeUtil.parseToDate;

/**
 * 告警服务实现类
 *
 * @author : Matrix [xhyrzldf@foxmail.com]
 * date  : 18-8-21
 */
@Slf4j
@Service
@Transactional(rollbackOn = Exception.class)
public class AlertServiceImpl implements AlertService {

    private static final String ALERT_NAME = "alert";
    @Autowired
    private AlertRepository alertRepo;
    @Autowired
    private FileReader fileReader;

    /**
     * 获取所有更新的报警数据,步骤如下
     * <li>1.到指定文件夹中读取还没处理过的文件(依据count值判断),并更新Count值</li>
     * <li>2.将数据转化为Alert类写入数据库</li>
     */
    @Override
    public void fetchUpdatedData() {
        //读取文件
        String jsonString = fileReader.readFileAndUnzip(COUNT_ADDRESS_ALERT, "pssp", ALERT_NAME);
        if (StringUtils.isEmpty(jsonString)) {
            log.info("[pssp] 本次更新采集没有要更新的数据");
            return;
        }

        List<Alert> updatedData = null;
        try {
            Alert[] alerts = JacksonUtil.readValue(jsonString, Alert[].class);
            updatedData = Arrays.asList(alerts);
        } catch (Exception e) {
            log.info("[pssp] 本次更新转换数据时发生异常 : {}", e.toString());
            e.printStackTrace();
        }

        //写入数据
        if (CollectionUtils.isEmpty(updatedData)) {
            log.info("[pssp] 本次更新采集没有要更新的数据");
        } else {
            log.info("[pssp] 采集完成,本次采集了 {} 条数据,数据写入本机数据库与Es数据库", updatedData.size());
//            List<Alert> savedUpdatedData = alertRepo.save(updatedData);
            log.info("[pssp] 数据库写入数据完成，本次写入 {} 条数据", updatedData.size());
            //完毕之后删除对应的数据
            fileReader.deleteFile(COUNT_ADDRESS_ALERT, "pssp", ALERT_NAME);
        }


    }

    /**
     * 查询所有数据
     */
    @Override
    public List<Alert> findAllData() {
        return alertRepo.findAll();
    }

    @Override
    public Page<Alert> findAllDataBetweenTime(LocalDate startTime, LocalDate endTime, Pageable pageable) {
        LocalDateTime maxEndTime = endTime.atTime(LocalTime.MAX);
        return alertRepo.findBetweenTime(
                LocalToDate(startTime),
                LocalToDate(maxEndTime),
                pageable);
    }

    /**
     * 查询指定年月
     * 例如查询 2018年7月的数据 构造出2018-07-01 与 2018-08-00这样的两端字符串
     * 然后使用between查询即可,这样可以利用到数据库的索引
     */
    @Override
    public List<Alert> findAllDataByMonth(int year, int month) {
        LocalDate startMonth = LocalDate.of(year, month, 1);
        LocalDate nextMonth = startMonth.plusMonths(1);

        return alertRepo.findBetweenTime(LocalToDate(startMonth), LocalToDate(nextMonth));
    }

    /**
     * 查询指定年月日的告警数据
     */
    @Override
    public List<Alert> findAllDataByDate(int year, int month, int day) throws Exception {
        Date specDate = parseToDate(year, month, day);
        return alertRepo.findByTmFetch(specDate);
    }

    /**
     * 查询最近指定天数内的告警数据
     * 等同于查询 当前时间-指定天数时间 之后的所有数据
     */
    @Override
    public List<Alert> findRecentDaysData(int days) {
        LocalDate daysAgo = LocalDate.now()
                .minusDays(days);
        log.info("[pssp]查询日期在 {} 之后的数据", LocalToDate(daysAgo));
        return alertRepo.findAfterTime(LocalToDate(daysAgo));
    }

    /**
     * 查询最近指定天数内的告警数据
     *
     * @param specifiedTime 指定时间 精确到秒
     * @param days          指定最近几天的时间
     * @return 报警数据
     */
    @Override
    public List<Alert> findRecentDaysData(LocalDateTime specifiedTime, int days) {
        LocalDateTime daysAgo = specifiedTime
                .minusDays(days);
        log.info("[pssp]查询日期在 {} 之后的数据", LocalToDate(daysAgo));
        return alertRepo.findAfterTime(LocalToDate(daysAgo));
    }


    /**
     * 根据主键id集合查询alert对象
     *
     * @param primaryIds 主键id集合
     * @return 报警集合
     */
    @Override
    public List<Alert> findAll(List<Long> primaryIds) {
        return alertRepo.findAll(primaryIds);
    }


    @Override
    public Page<Alert> findAllAfterId(Long primaryId, Pageable pageable) {
        //check id
        Alert alert = alertRepo.findOne(primaryId);
        if (Objects.isNull(alert)) {
            throw new InvalidParameterException("提供的primaryId并不正确,请提供实际存在的primaryId");
        }
        return alertRepo.findByPrimaryIdAfter(primaryId, pageable);
    }


    /**
     * 查询出报警数据的最早时间点.
     *
     * @return 最早的一条报警数据的时间
     */
    @Override
    public Date findEarliestTime() {
        return alertRepo.findFirstByOrderByTmFetch().getTmFetch();
    }

    /*  以下为私有方法区域 */
}
