package com.zjty.fp.api.pssp.subject.service.impl;

import com.zjty.fp.api.misc.utils.FileReader;
import com.zjty.fp.api.misc.utils.JacksonUtil;
import com.zjty.fp.api.pssp.subject.entity.location.Website;
import com.zjty.fp.api.pssp.subject.repository.location.WebsiteRepository;
import com.zjty.fp.api.pssp.subject.service.WebsiteService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.zjty.fp.api.misc.entity.PsspCount.COUNT_ADDRESS_ALERT;
import static com.zjty.fp.api.misc.entity.PsspCount.COUNT_ADDRESS_WEBSITE;

/**
 * fusion-platform.
 *
 * @author : Matrix [xhyrzldf@gmail.com]
 * 18-9-6 .
 */
@Slf4j
@Service
@Transactional(rollbackOn = Exception.class)
public class WebsiteServiceImpl implements WebsiteService {

    private static final String WEBSITE_NAME = "website";
    @Autowired
    private WebsiteRepository websiteRepository;
    @Autowired
    private FileReader fileReader;

    /**
     * 获取所有更新的网站数据，步骤如下
     * <li>1.到指定文件夹中读取还没处理过的文件(依据count值判断)</li>
     * <li>2.将数据转化为Alert类写入数据库</li>
     * <li>3.更新Count值</li>
     */
    @Override
    public void fetchUpdatedData() {
        //读取文件
        String jsonString = fileReader.readFileAndUnzip(COUNT_ADDRESS_WEBSITE, "pssp", WEBSITE_NAME);
        List<Website> updatedData = null;
        try {
            Website[] alerts = JacksonUtil.readValue(jsonString, Website[].class);
            updatedData = Arrays.asList(alerts);
        } catch (Exception e) {
            log.info("[pssp] 本次更新转换数据时发生异常 : {}",e.toString());
        }

        //写入数据
        if (CollectionUtils.isEmpty(updatedData)) {
            log.info("[pssp] 本次网站更新采集没有要更新的数据");
        } else {
            log.info("[pssp] 网站数据采集完成,本次采集了 {}条数据,数据写入本机数据库", updatedData.size());
            websiteRepository.save(updatedData);
            log.info("[pssp]异步网站更新数据写入完成,本次写入 {} 条数据", updatedData.size());
            //完毕之后删除对应的数据
            fileReader.deleteFile(COUNT_ADDRESS_WEBSITE, "pssp", WEBSITE_NAME);
        }



    }

    @Override
    public List<Website> findAllData() {
        return websiteRepository.findAll();
    }


    @Override
    public long getCount() {
        return websiteRepository.count();
    }

    /**
     * 网站数据以字典的形式返回
     *
     * @return key : id ,value : {@link Website}
     */
    @Override
    public Map<Long, Website> findDictMap() {
        return websiteRepository.findAll()
                .stream()
                .collect(Collectors
                        .toMap(Website::getId, Function.identity()));
    }
}
