package com.zjty.fp.api.union.base.exceptions;

import com.google.common.collect.ImmutableMap;
import com.zjty.fp.api.misc.utils.DateTimeUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

/**
 * <p>Description : fusion-platform
 * <p>Date : 2018/9/3 15:56
 * <p>@author : Mcj
 */

@SuppressWarnings("PlaceholderCountMatchesArgumentCount")
@RestControllerAdvice
@Slf4j
public class ExceptionHandlers {

    @ExceptionHandler(value = HttpRequestMethodNotSupportedException.class)

    public ResponseEntity httpRequestMethodNotSupportedExceptionHandler(Exception e) {
        log.error("抓取Http请求不支持异常:{}", e);
        return new ResponseEntity<>(ImmutableMap.of("message", "该接口不接受这样的Http类型访问,详情为" + e.getMessage()), HttpStatus.METHOD_NOT_ALLOWED);
    }

    @ExceptionHandler(value = NullPointerException.class)
    public ResponseEntity nullPointExceptionHandler(Exception e) {
        log.error("抓取空指针异常:{}", e);
        return new ResponseEntity<>(ImmutableMap.of("message", "空指针异常,详情为" + e.getMessage()), HttpStatus.EXPECTATION_FAILED);
    }

    @ExceptionHandler(value = DateTimeParseException.class)
    public ResponseEntity dateTimeParseExceptionHandler(DateTimeParseException e) {
        log.error("抓取时间类型转换异常异常:{}", e);
        return new ResponseEntity<>(ImmutableMap.of(
                "timestamp", LocalDateTime.now().format(DateTimeFormatter.ofPattern(DateTimeUtil.FMT_yyyyMMddHHmmss)),
                "status", HttpStatus.BAD_REQUEST.value(),
                "error", "时间转换异常,请提供正确的时间格式",
                "message", e.getMessage()), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler(value = Exception.class)
    public ResponseEntity exceptionHandler(Exception e) {
        log.error("抓取全局异常:{}", e);
        return new ResponseEntity<>(ImmutableMap.of(
                "error","通用错误",
                "message", e.getMessage()), HttpStatus.BAD_REQUEST);
    }


    /**
     * 自定义异常处理
     */
    @ExceptionHandler(value = CustomException.class)
    public ResponseEntity customExceptionHandler(CustomException e) {
        log.warn("抓取自定义异常:{}", e);
        return new ResponseEntity<>(e.getMessage(), HttpStatus.BAD_REQUEST);
    }
}
