package org.matrix.actuators.variable;

import com.alibaba.fastjson.JSON;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.matrix.actuators.Actuator;
import org.matrix.actuators.env.EnvironmentActuator;
import org.matrix.actuators.httpclient.HttpClientActuator;
import org.matrix.actuators.httpclient.HttpRequestDetail;
import org.matrix.actuators.httpclient.HttpResponseDetail;
import org.matrix.actuators.usecase.CaseActuator;
import org.matrix.datasource.DataSourceDTO;
import org.matrix.datasource.IDataSourceService;
import org.matrix.entity.DynamicVariable;
import org.matrix.entity.TestCaseBTO;
import org.matrix.enums.DynamicVarType;
import org.matrix.exception.GlobalException;
import org.matrix.service.IConnectService;
import org.matrix.service.IDynamicVariableService;
import org.matrix.service.ITestDataService;
import org.matrix.socket.queue.LogQueueRuntime;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.matrix.enums.DynamicVarType.*;
import static org.matrix.enums.ModuleType.SQL_ACTUATOR;
import static org.springframework.util.CollectionUtils.isEmpty;

/**
 * VarActuator.
 *
 * @author Matrix <xhyrzldf@gmail.com>
 * @since 2022/6/27 at 13:55
 * Suffering is the most powerful teacher of life.
 */
@Component
@Slf4j
@AllArgsConstructor
public class VarActuator implements Actuator {

    /**
     * 用于正则找出形如${id} 或者 ${id}[1] 这样的SQL变量
     */
    public static final String DYNAMIC_VAR_EXP = "\\$\\{(\\w*)}\\[?(\\d*)]?";

    //language=RegExp
    /**
     * 用于replaceAll的正则，目的是替换掉形如${id} 或者是 ${id}[1] 这样的SQL变量
     */
    public static final String REPLACE_VAR_EXP = "\\$\\{%s}(\\[\\w*])?";
    private final IDynamicVariableService varService;
    private final ITestDataService dataService;
    private final HttpClientActuator httpActuator;
    private final CaseActuator caseActuator;
    private final JdbcTemplate jdbcTemplate;
    private final EnvironmentActuator envActuator;
    private final IConnectService connectService;
    private final IDataSourceService dataSourceService;


    /**
     * 全局变量池, key = 雪花id ,代表唯一的执行次序。
     * value = 全局变量池
     */
    private Map<String,Map<String,String>> GLOBAL_VAR_MAP = new HashMap<>();

    /**
     * 临时表量池, key = 雪花id + 套件主键id , value = 临时表量池
     */
    private Map<String,Map<String,String>> TEMP_VAR_MAP = new HashMap<>();

    /**
     * 保存全局变量
     * @param key 雪花执行id
     * @param value 全局变量池
     * @return 是否保存成功
     */
    public boolean saveGlobalVar(String key, String value) {
        if (StringUtils.isBlank(key) || StringUtils.isBlank(value)) {
            return false;
        }
        Map<String, String> varMap = GLOBAL_VAR_MAP.computeIfAbsent(key, k -> new HashMap<>(64));
        varMap.put(value, value);
        return true;
    }

    /**
     * 保存临时表量
     * @param key 雪花执行id + 套件主键id
     * @param value 临时表量池
     * @return 是否保存成功
     */
    public boolean saveTempVar(String key, String value) {
        if (StringUtils.isBlank(key) || StringUtils.isBlank(value)) {
            return false;
        }
        Map<String, String> varMap = TEMP_VAR_MAP.computeIfAbsent(key, k -> new HashMap<>(64));
        varMap.put(value, value);
        return true;
    }



    /**
     * <h2>解析给定的动态变量ByName</h2>
     * <p>1. 在临时变量池中 寻找 是否存在 userid 的变量，如果有就取它的值，如果没有就第二步
     * <p>2. 在全局变量池中 寻找 是否存在 userid 的变量，如果有就取它的值，如果没有就第三步
     * <p>3. 在变量数据库中寻找是否存在名为 userid的变量，如果有就进行计算并获取它的数值,如果没有就进行第四部
     * <p>4. 抛出找不到 名为 userid 变量的异常
     *
     * @param varNameString 变量名,形如${id}或者是 ${id}[1]这样的字符串,默认下标为0
     * @param projectId     该变量所在的项目ID
     * @return 变量递归解析后的值
     */
    public String parseVarByName(String varNameString, Long envId, Long projectId) {
        // 解析变量字符串, 分离出变量名和下标
        LogQueueRuntime.addNewLog(this.getClass(), SQL_ACTUATOR, String.format("初始化变量,变量名=%s,环境id=%d,项目id=%d", varNameString, envId, projectId));
        List<VarIndexTuple> varList = extractVarNameIndex(varNameString);
        if (varList.size() == 1) {
            VarIndexTuple tuple = varList.get(0);
            // 在临时变量池中 寻找 是否存在 userid 的变量，如果有就取它的值，如果没有就第二步
//            String varValue = getVarValueFromTempMap(tuple.getVarName());

            DynamicVariableObj variable = varService.getByName(tuple.getVarName(), projectId)
                    .map(DynamicVariableObj::var2Obj)
                    .orElseThrow(() -> new GlobalException(String.format("没有找到project id = %d 且 name = %s 的动态变量", projectId, tuple.getVarName())));
            return parseVar(envId, tuple.getIndex(), projectId, variable);
        } else {
            throw new GlobalException(String.format("varNameString 参数请一次只输入一个动态变量(形如 ${id})! 你的提供的参数为: %s , 解析后的变量列表是 %s", varNameString, varList));
        }
    }


    /**
     * 给定一个动态变量对象,依据不同的变量类型调用不同的解析Handler
     * 返回解析后的字符串结果
     *
     * @param envId      环境ID - 环境共享变量需要
     * @param takenIndex 使用的结果集下标,递归传递
     * @param projectId  项目id，防止SQL表达式中带的嵌合动态变量名字重复
     * @param dynamicVar {@link DynamicVariable}
     * @return 变量递归解析后的值
     */
    private String parseVar(Long envId, Integer takenIndex, Long projectId, DynamicVariableObj dynamicVar) {
        DynamicVarType varType = dynamicVar.getType();
        LogQueueRuntime.addNewLog(this.getClass(), SQL_ACTUATOR, String.format(
                "当前解析的动态变量名 = %s , 变量详情 = %s , 变量类型 = %s ," +
                        " takenField = %s , 备注 = %s",
                dynamicVar.getName(), dynamicVar.getDetail(), dynamicVar.getType().getDes(), dynamicVar.getTakenField(), dynamicVar.getRemark()));
        // 依据变量类型的不同调用不同的Handler
        if (varType == CONSTANT_VARIABLE) {
            return dynamicVar.getDetail();
        } else if (varType == SQL_VARIABLE) {
            return sqlVarHandler(envId, takenIndex, projectId, dynamicVar);
        } else if (varType == HTTP_VARIABLE) {
            return httpVarHandler(envId, projectId, dynamicVar);
        } else if (varType == CASE_VARIABLE) {
            return caseHandler(envId, projectId, dynamicVar);
        } else {
            throw new GlobalException("不支持的动态变量类型: " + varType);
        }
    }


    /**
     * 将含有动态变量的字符串语句中的形如{$id}动态变量语句给找到
     * 如果使用者写了诸如{$id}[1]的下标,则会使用下标值,否则使用默认值0
     *
     * @param dynamicString SQL字符串或者动态变量
     * @return SQL动态变量对象, 分别找出其中的变量名与对应的下标序列值(默认为0)
     */
    public List<VarIndexTuple> extractVarNameIndex(String dynamicString) {
        Pattern pattern = Pattern.compile(DYNAMIC_VAR_EXP);
        Matcher matcher = pattern.matcher(dynamicString);
        List<VarIndexTuple> sqlRegObjects = new ArrayList<>();
        // 如果使用者写了诸如{$id}[1]的下标,则会使用下标值,否则使用默认值0
        while (matcher.find()) {
            if (matcher.groupCount() != 2) {
                throw new GlobalException("给定的动态变量/SQL表达式 不符合语法,给定的值是: " + dynamicString);
            }
            String varName = matcher.group(1);
            int index = StringUtils.isEmpty(matcher.group(2)) ? 0 : Integer.parseInt(matcher.group(2));
            sqlRegObjects.add(new VarIndexTuple(varName, index));
        }
        return sqlRegObjects;
    }

    /**
     * 解析并运行SQL语句，可以包含别的SQL变量,例如 'select * from user where user.id = ${user_id}'
     *
     * @param sqlDetail sql的poolId与sql语句的对象
     * @param envId     环境id
     * @param projectId 项目id
     * @return 运行SQL后取完数值的结果, 是一张数据表, 以LIST<MAP>的形式呈现
     */
    public List<Map<String, Object>> parseSql(SqlExpDetail sqlDetail, Long envId, Long projectId) {
        // 找到第一层的SQL变量，递归替换
        String sqlExp = sqlDetail.getSqlExp();
        Long connectId = sqlDetail.getPoolId();
        List<VarIndexTuple> varList = extractVarNameIndex(sqlExp);
        for (VarIndexTuple sqlReg : varList) {
            sqlExp = sqlExp.replaceAll(sqlReg.getVarName(), parseVarByName(sqlReg.getVarName(), envId, projectId));
        }

        // 运行SQL
        DataSourceDTO dataSourceDTO = Optional.of(connectService.getById(connectId))
                .orElseThrow(() -> new GlobalException(String.format("没有找到id = %d 的连接池connect对象", connectId)))
                .toDataSourceDTO();

        // 替换环境共享变量
        sqlExp = envActuator.replaceEnvVar(sqlExp, envId);

        // 校验dynamicVar里的detail是否是可以直接执行的SQL
        boolean allVarParsed = extractVarNameIndex(sqlExp).size() == 0;
        if (allVarParsed) {
            // 切换数据源,执行SQL,获取数值
            Set<String> dataSources = dataSourceService.switchDataSource(dataSourceDTO);
            log.info("当前存在的数据源 {}", dataSources);
            List<Map<String, Object>> resultMap = jdbcTemplate.queryForList(sqlExp);
            dataSourceService.switchMainDataSource();
            return resultMap;
        } else {
            throw new GlobalException(String.format("SQL解析异常,SQL语句为%s,连接池id为%d", sqlDetail.getSqlExp(), sqlDetail.getPoolId()));
        }
    }

    /**
     * SQL表达式处理器
     *
     * @param envId      环境ID - 环境共享变量需要
     * @param projectId  项目id，防止SQL表达式中带的嵌合动态变量名字重复
     * @param variableObj {@link DynamicVariable}
     * @return 解析后的字符串
     */
    @SuppressWarnings("DuplicatedCode")
    private String sqlVarHandler(Long envId, Integer takenIndex, Long projectId, DynamicVariableObj variableObj) {

        // 首先替换掉#{id}这类的共享变量(如果有的话)
        String sqlExp = variableObj.getSqlExpDetail().getSqlExp();
        if (envId != null && envId > 0) {
            sqlExp = envActuator.replaceEnvVar(sqlExp, envId);
        }

        String detail = variableObj.getDetail();
        List<VarIndexTuple> dynamicVarList = extractVarNameIndex(detail);
        // 解析SQL表达式，判断是可以直接执行的SQL还是需要再递归解析动态变量
        if (!isEmpty(dynamicVarList)) {
            // 如果还存在动态变量,则继续递归解析
            for (VarIndexTuple sqlRegular : dynamicVarList) {
                DynamicVariableObj recVar = varService.getByName(sqlRegular.getVarName(), projectId)
                        .map(DynamicVariableObj::var2Obj)
                        .orElseThrow(() -> new GlobalException(
                                String.format("没有找到项目id = %d 下,name = %s的变量", projectId, sqlRegular.getVarName())));
                String calculatedValue = parseVar(envId, sqlRegular.getIndex(), projectId, recVar);
                LogQueueRuntime.addNewLog(this.getClass(), SQL_ACTUATOR, String.format("正在进行SQL变量计算 变量名: %s 计算后的替换值: %s", sqlRegular.getVarName(), calculatedValue));
                sqlExp = sqlExp.replaceAll(String.format(REPLACE_VAR_EXP, sqlRegular.getVarName()), calculatedValue);
                variableObj.getSqlExpDetail().setSqlExp(sqlExp);
            }
        }
        // 执行SQL
        List<Map<String, Object>> resultList = runSqlDirectly(variableObj, envId);
        //正则
        return takenResultString(resultList, variableObj.getTakenField(), Math.toIntExact(takenIndex));
    }

    /**
     * 递归的运行一段SQL语句，可以解析出其中的环境变量(#{id}),动态变量(${id})
     *
     * @param projectId 用于确定寻找哪些动态变量的id
     * @param envId     用于确定使用的环境变量池的id,如果envId = null,则代表无需替换环境变量
     * @param connectId 用于确定使用哪个数据源
     * @return 运行SQL后取完数值的结果, 是一张数据表, 以LIST<MAP>的形式呈现
     */
    @SuppressWarnings("DuplicatedCode")
    public List<Map<String, Object>> runSqlRec(String sqlExp, Long projectId, Long envId, Long connectId) {
        // 替换环境表达式
        if (envId != null && envId > 0) {
            sqlExp = envActuator.replaceEnvVar(sqlExp, envId);
        }
        List<VarIndexTuple> dynamicVarList = extractVarNameIndex(sqlExp);
        // 解析SQL表达式，判断是可以直接执行的SQL还是需要再递归解析动态变量
        if (!isEmpty(dynamicVarList)) {
            // 如果还存在动态变量,则继续递归解析
            for (VarIndexTuple sqlRegular : dynamicVarList) {
                DynamicVariableObj recVar = varService.getByName(sqlRegular.getVarName(), projectId)
                        .map(DynamicVariableObj::var2Obj)
                        .orElseThrow(() -> new GlobalException(
                                String.format("没有找到项目id = %d 下,name = %s的变量", projectId, sqlRegular.getVarName())));
                String calculatedValue = parseVar(envId, sqlRegular.getIndex(), projectId, recVar);
                LogQueueRuntime.addNewLog(this.getClass(), SQL_ACTUATOR, String.format("正在进行SQL变量计算 变量名: %s 计算后的替换值: %s", sqlRegular.getVarName(), calculatedValue));
                sqlExp = sqlExp.replaceAll(String.format(REPLACE_VAR_EXP, sqlRegular.getVarName()), calculatedValue);
            }
        }
        // 切换数据源，执行SQL
        DataSourceDTO dataSourceDTO = Optional.of(connectService.getById(connectId))
                .orElseThrow(() -> new GlobalException(String.format("没有找到id = %d 的连接池connect对象", connectId)))
                .toDataSourceDTO();

        // 切换数据源,执行SQL,获取数值
        LogQueueRuntime.addNewLog(getClass(), SQL_ACTUATOR, String.format("当前执行的SQL语句: %s, 使用的数据源: %s", sqlExp, dataSourceDTO));
        dataSourceService.switchDataSource(dataSourceDTO);
        log.info("当前使用的的数据源 {}", dataSourceService.peek());
        List<Map<String, Object>> resultMap = jdbcTemplate.queryForList(sqlExp);
        dataSourceService.switchMainDataSource();
        return resultMap;
    }

    /**
     * 用例处理器
     *
     * @param envId      环境ID - 环境共享变量需要
     * @param projectId  项目id，防止SQL表达式中带的嵌合动态变量名字重复
     * @param dynamicVar {@link DynamicVariable}
     * @return 解析后的字符串
     */
    private String caseHandler(Long envId, Long projectId, DynamicVariable dynamicVar) {
        long caseDataId = Long.parseLong(dynamicVar.getDetail());
        TestCaseBTO caseBTO = dataService.toCaseBTO(caseDataId);
        HttpResponseDetail responseDetail = (HttpResponseDetail) caseActuator.executeTestCase(caseBTO, envId, projectId)
                .getBaseTestCaseResponseDetail();
        return responseDetail.getResponseBody();
    }

    /**
     * 递归的运行一段SQL语句，可以解析出其中的环境变量(#{id}),动态变量(${id})
     *
     * @param projectId 用于确定寻找哪些动态变量的id
     * @param envId     用于确定使用的环境变量池的id,如果envId = null,则代表无需替换环境变量
     * @return 运行SQL后取完数值的结果, 是一张数据表, 以LIST<MAP>的形式呈现
     */
    @SuppressWarnings("DuplicatedCode")
    public List<Map<String, Object>> runActuatorSqlRec(String sqlExp, Long projectId, Long envId) {
        // 替换环境表达式
        if (envId != null && envId > 0) {
            sqlExp = envActuator.replaceEnvVar(sqlExp, envId);
        }
        List<VarIndexTuple> dynamicVarList = extractVarNameIndex(sqlExp);
        // 解析SQL表达式，判断是可以直接执行的SQL还是需要再递归解析动态变量
        if (!isEmpty(dynamicVarList)) {
            // 如果还存在动态变量,则继续递归解析
            for (VarIndexTuple sqlRegular : dynamicVarList) {
                DynamicVariableObj recVar = varService.getByName(sqlRegular.getVarName(), projectId)
                        .map(DynamicVariableObj::var2Obj)
                        .orElseThrow(() -> new GlobalException(
                                String.format("没有找到项目id = %d 下,name = %s的变量", projectId, sqlRegular.getVarName())));
                String calculatedValue = parseVar(envId, sqlRegular.getIndex(), projectId, recVar);
                LogQueueRuntime.addNewLog(this.getClass(), SQL_ACTUATOR, String.format("正在进行SQL变量计算 变量名: %s 计算后的替换值: %s", sqlRegular.getVarName(), calculatedValue));
                sqlExp = sqlExp.replaceAll(String.format(REPLACE_VAR_EXP, sqlRegular.getVarName()), calculatedValue);
            }
        }
        return jdbcTemplate.queryForList(sqlExp);
    }

    /**
     * HTTP请求处理器
     *
     * @param envId      环境ID - 环境共享变量需要
     * @param projectId  项目id，防止SQL表达式中带的嵌合动态变量名字重复
     * @param dynamicVar {@link DynamicVariable}
     * @return 解析后的字符串
     */
    private String httpVarHandler(Long envId, Long projectId, DynamicVariable dynamicVar) {
        HttpRequestDetail httpRequestDetail = JSON.parseObject(dynamicVar.getDetail(), HttpRequestDetail.class);
        return httpActuator.sendHttpRequest(httpRequestDetail, envId, projectId).getResponseBody();
    }

    /**
     * @param resultList 初始查询的结果集(表)
     * @param takenField 要取用的字段(列)
     * @param index      要取用的行数,从0开始算第一行
     * @return 取到的字段的结果
     * @throws GlobalException 如果找不到对应行列的数据(index与takenField),会抛出这个异常,建议把上层的SQL语句也返回到信息里
     */
    private String takenResultString(List<Map<String, Object>> resultList, String takenField, int index) throws GlobalException {
        Map<String, Object> resultMap = Optional.of(resultList.get(index))
                .orElseThrow(() -> new GlobalException(String.format("没有在 %s 中找到第 %d 行 的数据",
                        JSON.toJSON(resultList), index + 1)));

        return Optional.of(resultMap.get(takenField))
                .orElseThrow(() -> new GlobalException(String.format("%s 数据集中不存在属性名为 %s 的字段",
                        JSON.toJSON(resultMap), takenField))).toString();
    }

    /**
     * 直接执行一段SQL，该SQL表达式必须要是可以直接执行的语句
     *
     * @param variableObj 动态变量对象(必须是可以直接执行SQL 例`select * from user where id = 5`)
     * @param envId      环境ID
     * @return 运行SQL后取完数值的结果, 是一张数据表, 以LIST<MAP>的形式呈现
     */
    private List<Map<String, Object>> runSqlDirectly(DynamicVariableObj variableObj, Long envId) {
        String sqlExp = variableObj.getSqlExpDetail().getSqlExp();
        Long connectId = variableObj.getSqlExpDetail().getPoolId();
        DataSourceDTO dataSourceDTO = Optional.of(connectService.getById(connectId))
                .orElseThrow(() -> new GlobalException(String.format("没有找到id = %d 的连接池connect对象", connectId)))
                .toDataSourceDTO();

        // 替换环境共享变量
        sqlExp = envActuator.replaceEnvVar(sqlExp, envId);

        // 校验dynamicVar里的detail是否是可以直接执行的SQL
        if (variableObj.getType() == SQL_VARIABLE && extractVarNameIndex(sqlExp).size() == 0) {
            // 切换数据源,执行SQL,获取数值
            LogQueueRuntime.addNewLog(this.getClass(), SQL_ACTUATOR, String.format("当前执行的SQL语句: %s, 使用的数据源: %s", sqlExp, dataSourceDTO));
            Set<String> dataSources = dataSourceService.switchDataSource(dataSourceDTO);
            log.info("当前存在的数据源 {}", dataSources);
            List<Map<String, Object>> resultMap = jdbcTemplate.queryForList(sqlExp);
            dataSourceService.switchMainDataSource();
            return resultMap;
        } else {
            throw new GlobalException(String.format("动态变量的类型必须要是SQL类型且DETAIL内的SQL语句必须为元SQL语句,当前type : %s , 当前detail : %s",
                    variableObj.getType(), sqlExp));
        }

    }

}
