package org.matrix.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.matrix.entity.MouldDoc;
import org.matrix.entity.MouldGrouping;
import org.matrix.enums.SwaggerDataType;
import org.matrix.enums.SyncStatus;
import org.matrix.exception.GlobalException;
import org.matrix.mapper.MouldDocMapper;
import org.matrix.mapper.MouldGroupingMapper;
import org.matrix.service.IMouldDocService;
import org.matrix.vo.ImportMouldDoc;
import org.matrix.vo.MouldVo;
import org.matrix.vo.StatusCode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author mruny
 * @create 2022/7/18 11:29:15
 */
@Service
public class MouldDocServiceImpl extends ServiceImpl<MouldDocMapper, MouldDoc> implements IMouldDocService {

    @Autowired
    private MouldDocMapper mouldDocMapper;

    @Autowired
    private MouldGroupingMapper mouldGroupingMapper;

    @Override
    public StatusCode importMouldDoc(List<MouldVo> mouldVos, Long projectId, int pattern, Long mouldId) {
        StatusCode statusCode;
        switch (pattern) {
            //1. 同名覆盖
            case ImportMouldDoc.NAME_OVERWRITE:
                statusCode = nameOverwrite(mouldVos, projectId, mouldId);
                break;
            //2. 同名同组覆盖
            case ImportMouldDoc.NAME_GROUPING_OVERWRITE:
                statusCode = nameGroupingOverwrite(mouldVos, projectId, mouldId);
                break;
            //3. 同名不覆盖
            case ImportMouldDoc.NOT_NAME_OVERWRITE:
                statusCode = notNameOverwrite(mouldVos, projectId, mouldId);
                break;
            //4. 同名同组不覆盖
            case ImportMouldDoc.NOT_NAME_GROUPING_OVERWRITE:
                statusCode = notNameGroupingOverwrite(mouldVos, projectId, mouldId);
                break;
            //5. 同时存在
            case ImportMouldDoc.COEXIST:
                statusCode = coexist(mouldVos, projectId, mouldId);
                break;
            default:
                throw new GlobalException("请选择正确的导入类型");
        }
        return statusCode;
    }

    public List<MouldDoc> getMouldDoc(List<MouldVo> mouldVos) {
        List<MouldDoc> mouldDocs = new ArrayList<>();
        if (!CollectionUtils.isEmpty(mouldVos)) {
            for (MouldVo mouldVo : mouldVos) {
                MouldDoc mouldDoc = new MouldDoc();
                mouldDoc.setObjName(mouldVo.getObjName());
//                mouldDoc.setAttributeOutVo(mouldVo.getAttributeOutVo());
                mouldDocs.add(mouldDoc);
            }
        }
        return mouldDocs;
    }

    /**
     * 1. 同名覆盖
     *
     * @param mouldVos  swagger导入的数据模型集合
     * @param projectId 项目id
     * @param mouldId   分组Id(导入时选择导入到哪个分组中)
     * @return 导入结果
     */
    public StatusCode nameOverwrite(List<MouldVo> mouldVos, Long projectId, Long mouldId) {
        int saveNum = 0;
        int updateNum = 0;
        int ignoreNum = 0;
        StatusCode statusCode = new StatusCode();
        statusCode.setSwaggerDataType(SwaggerDataType.MOULD_DOC);
        if (mouldVos.size() != 0) {
            List<MouldDoc> mouldDocs = getMouldDoc(mouldVos);
            List<Long> longList = getIdList(projectId);
            Long groupingId = getGroupingId(mouldId, projectId);
            for (MouldDoc mouldDoc : mouldDocs) {
                //将分组加到对象中
                mouldDoc.setMouldGroupingId(groupingId);
                mouldDoc.setProjectId(projectId);
                //找到可以覆盖的
                List<MouldDoc> mouldDocList = Optional.of(mouldDocMapper.selectList(Wrappers.lambdaQuery(MouldDoc.class)
                        .eq(MouldDoc::getObjName, mouldDoc.getObjName())
                        .eq(MouldDoc::getProjectId, projectId))).orElse(new ArrayList<>());
                //覆盖第一个
                if (mouldDocList.size() != 0) {
                    //如果存在就覆盖第一个
                    Long mouldDocId = mouldDocList.get(0).getId();
                    mouldDoc.setId(mouldDocId);
                    mouldDoc.setStatus(SyncStatus.UPDATE);
                    Integer integer = Optional.of(mouldDocMapper.updateById(mouldDoc)).orElseThrow(() -> new GlobalException("修改数据模型失败"));
                    updateNum = updateNum + integer;
                    longList.remove(mouldDoc.getId());
                } else {
                    mouldDoc.setStatus(SyncStatus.INSERT);
                    //如果不存在就直接添加
                    Integer integer = Optional.of(mouldDocMapper.insert(mouldDoc)).orElseThrow(() -> new GlobalException("添加数据模型失败"));
                    saveNum = saveNum + integer;
                }
            }
            updateStatusNo(longList);
        }
        statusCode.setSaveNum(saveNum);
        statusCode.setUpdateNum(updateNum);
        statusCode.setIgnoreNum(ignoreNum);
        return statusCode;
    }

    /**
     * 2. 同名同组覆盖
     *
     * @param mouldVos  swagger导入的数据模型集合
     * @param projectId 项目id
     * @param mouldId   分组Id(导入时选择导入到哪个分组中)
     * @return 导入结果
     */
    public StatusCode nameGroupingOverwrite(List<MouldVo> mouldVos, Long projectId, Long mouldId) {
        int saveNum = 0;
        int updateNum = 0;
        int ignoreNum = 0;
        StatusCode statusCode = new StatusCode();
        statusCode.setSwaggerDataType(SwaggerDataType.MOULD_DOC);
        if (mouldVos.size() != 0) {
            List<MouldDoc> mouldDocs = getMouldDoc(mouldVos);
            List<Long> longList = getIdList(projectId);
            Long groupingId = getGroupingId(mouldId, projectId);
            for (MouldDoc mouldDoc : mouldDocs) {
                //将新的分组存入到数据中
                mouldDoc.setProjectId(projectId);
                mouldDoc.setMouldGroupingId(groupingId);
                //找到可以覆盖的
                List<MouldDoc> mouldDocList = Optional.of(mouldDocMapper.selectList(Wrappers.lambdaQuery(MouldDoc.class)
                        .eq(MouldDoc::getObjName, mouldDoc.getObjName())
                        .eq(MouldDoc::getProjectId, projectId)
                        .eq(MouldDoc::getMouldGroupingId, mouldDoc.getMouldGroupingId()))).orElse(new ArrayList<>());
                //覆盖第一个
                if (mouldDocList.size() != 0) {
                    //如果存在就覆盖第一个
                    Long mouldDocId = mouldDocList.get(0).getId();
                    mouldDoc.setId(mouldDocId);
                    mouldDoc.setStatus(SyncStatus.UPDATE);
                    Integer integer = Optional.of(mouldDocMapper.updateById(mouldDoc)).orElseThrow(() -> new GlobalException("修改数据模型失败"));
                    updateNum = updateNum + integer;
                } else {
                    mouldDoc.setStatus(SyncStatus.INSERT);
                    //如果不存在就直接添加
                    Integer integer = Optional.of(mouldDocMapper.insert(mouldDoc)).orElseThrow(() -> new GlobalException("添加数据模型失败"));
                    saveNum = saveNum + integer;
                }
            }
            updateStatusNo(longList);
        }
        statusCode.setSaveNum(saveNum);
        statusCode.setUpdateNum(updateNum);
        statusCode.setIgnoreNum(ignoreNum);
        return statusCode;
    }

    /**
     * 3. 同名不覆盖
     *
     * @param mouldVos  swagger导入的数据模型集合
     * @param projectId 项目id
     * @param mouldId   分组Id(导入时选择导入到哪个分组中)
     * @return 导入结果
     */
    public StatusCode notNameOverwrite(List<MouldVo> mouldVos, Long projectId, Long mouldId) {
        int saveNum = 0;
        int updateNum = 0;
        int ignoreNum = 0;
        StatusCode statusCode = new StatusCode();
        statusCode.setSwaggerDataType(SwaggerDataType.MOULD_DOC);
        if (mouldVos.size() != 0) {
            List<MouldDoc> mouldDocs = getMouldDoc(mouldVos);
            List<Long> longList = getIdList(projectId);
            Long groupingId = getGroupingId(mouldId, projectId);
            for (MouldDoc mouldDoc : mouldDocs) {
                //将新的分组存入到数据中
                mouldDoc.setMouldGroupingId(groupingId);
                mouldDoc.setProjectId(projectId);
                //找到可以不做任何操作的
                List<MouldDoc> mouldDocList = Optional.of(mouldDocMapper.selectList(Wrappers.lambdaQuery(MouldDoc.class)
                        .eq(MouldDoc::getObjName, mouldDoc.getObjName())
                        .eq(MouldDoc::getProjectId, projectId))).orElse(new ArrayList<>());
                //忽略数+1
                if (mouldDocList.size() != 0) {
                    ignoreNum = ignoreNum + 1;
                } else {
                    //如果不存在就直接添加
                    Integer integer = Optional.of(mouldDocMapper.insert(mouldDoc)).orElseThrow(() -> new GlobalException("添加数据模型失败"));
                    saveNum = saveNum + integer;
                }
            }
            updateStatusNo(longList);
        }
        statusCode.setSaveNum(saveNum);
        statusCode.setUpdateNum(updateNum);
        statusCode.setIgnoreNum(ignoreNum);
        return statusCode;
    }

    /**
     * 4. 同名同组不覆盖
     *
     * @param mouldVos  swagger导入的数据模型集合
     * @param projectId 项目id
     * @param mouldId   分组Id(导入时选择导入到哪个分组中)
     * @return 导入结果
     */
    public StatusCode notNameGroupingOverwrite(List<MouldVo> mouldVos, Long projectId, Long mouldId) {
        int saveNum = 0;
        int updateNum = 0;
        int ignoreNum = 0;
        StatusCode statusCode = new StatusCode();
        statusCode.setSwaggerDataType(SwaggerDataType.MOULD_DOC);
        if (mouldVos.size() != 0) {
            List<MouldDoc> mouldDocs = getMouldDoc(mouldVos);
            List<Long> longList = getIdList(projectId);
            Long groupingId = getGroupingId(mouldId, projectId);
            for (MouldDoc mouldDoc : mouldDocs) {
                //将新的分组存入到数据中
                mouldDoc.setMouldGroupingId(groupingId);
                mouldDoc.setProjectId(projectId);
                //找到可以不做任何操作的
                List<MouldDoc> mouldDocList = Optional.of(mouldDocMapper.selectList(Wrappers.lambdaQuery(MouldDoc.class)
                        .eq(MouldDoc::getObjName, mouldDoc.getObjName())
                        .eq(MouldDoc::getProjectId, projectId)
                        .eq(MouldDoc::getMouldGroupingId, mouldDoc.getMouldGroupingId()))).orElse(new ArrayList<>());
                //忽略数+1
                if (mouldDocList.size() != 0) {
                    ignoreNum = ignoreNum + 1;
                    //上次忽略的
                    for (MouldDoc doc : mouldDocList) {
                        doc.setStatus(SyncStatus.IGNORE);
                        Integer integer = Optional.of(mouldDocMapper.updateById(doc)).orElseThrow(() -> new GlobalException("修改失败"));
                        ignoreNum = ignoreNum + integer;
                    }
                } else {
                    //如果不存在就直接添加
                    Integer integer = Optional.of(mouldDocMapper.insert(mouldDoc)).orElseThrow(() -> new GlobalException("添加数据模型失败"));
                    saveNum = saveNum + integer;
                }
            }
            updateStatusNo(longList);
        }
        statusCode.setSaveNum(saveNum);
        statusCode.setUpdateNum(updateNum);
        statusCode.setIgnoreNum(ignoreNum);
        return statusCode;
    }

    /**
     * 5. 同名时保留两者
     *
     * @param mouldVos  swagger导入的数据模型集合
     * @param mouldId   分组Id(导入时选择导入到哪个分组中)
     * @param projectId 项目id
     * @return 导入结果
     */
    public StatusCode coexist(List<MouldVo> mouldVos, Long projectId, Long mouldId) {
        int saveNum = 0;
        int updateNum = 0;
        int ignoreNum = 0;
        StatusCode statusCode = new StatusCode();
        statusCode.setSwaggerDataType(SwaggerDataType.MOULD_DOC);
        if (mouldVos.size() != 0) {
            List<MouldDoc> mouldDocs = getMouldDoc(mouldVos);
            //当前项目中的所有数据模型
            List<MouldDoc> mouldDocList = Optional.ofNullable(mouldDocMapper.selectList(Wrappers.lambdaQuery(MouldDoc.class)
                    .eq(MouldDoc::getProjectId, projectId))).orElse(new ArrayList<>());
            //按照名称分组
            Map<String, List<MouldDoc>> collect = mouldDocList.stream()
                    .filter(mouldDoc -> mouldDoc != null && mouldDoc.getObjName() != null && !"".equals(mouldDoc.getObjName()))
                    .collect(Collectors.groupingBy(MouldDoc::getObjName));
            //存在的不同名称的数据模型(如果同名取第一个)
            List<MouldDoc> list = new ArrayList<>();
            //每组取第一个
            if (!CollectionUtils.isEmpty(collect)) {
                Set<String> objNames = collect.keySet();
                if (!CollectionUtils.isEmpty(objNames)) {
                    for (String objName : objNames) {
                        MouldDoc mouldDoc = collect.get(objName).get(0);
                        list.add(mouldDoc);
                    }
                }
            }
            Long groupingId = getGroupingId(mouldId, projectId);
            //存数据,并且将数据库中同名的数据模型修改未未同步
            for (MouldDoc mouldDoc : mouldDocs) {
                mouldDoc.setMouldGroupingId(groupingId);
                mouldDoc.setProjectId(projectId);
                mouldDoc.setStatus(SyncStatus.COPY);
                Integer integer = Optional.of(mouldDocMapper.insert(mouldDoc)).orElseThrow(() -> new GlobalException("添加数据模型失败"));
                saveNum = saveNum + integer;
                //操作数据库中存在的数据模型
                for (MouldDoc doc : list) {
                    //如果同名，就将库存中的同名数据模型修改为未同步
                    if (mouldDoc.getObjName().equals(doc.getObjName())) {
                        list.remove(doc);
                        doc.setStatus(SyncStatus.NO);
                        mouldDocMapper.updateById(doc);
                        break;
                    }
                }
            }
            //剩下的就是当前存在，但是导入的数据中不存在的
            if (list.size() != 0) {
                for (MouldDoc mouldDoc : list) {
                    mouldDoc.setStatus(SyncStatus.OTHER);
                    mouldDocMapper.updateById(mouldDoc);
                }
            }
        }
        statusCode.setSaveNum(saveNum);
        statusCode.setUpdateNum(updateNum);
        statusCode.setIgnoreNum(ignoreNum);
        return statusCode;
    }

    /**
     * 获取分组的主键id
     *
     * @param mouldId   父级分组id
     * @param projectId 项目id
     * @return 当前加入的Schemas分组的主键id
     */
    private Long getGroupingId(Long mouldId, Long projectId) {
        Long id;
        //当前是不是0无所谓，只需要根据mouldId查出当前分组的主键id就可以，如果存在取第一个，如果不存在就创建
        List<MouldGrouping> mouldGroupings = Optional.of(mouldGroupingMapper.selectList(Wrappers.lambdaQuery(MouldGrouping.class)
                .eq(MouldGrouping::getMouldId, mouldId)
                .eq(MouldGrouping::getProjectId, projectId)
                .eq(MouldGrouping::getName, "Schemas"))).orElse(new ArrayList<>());
        //如果分组存在,就直接将数据模行放入到分组中
        if (mouldGroupings.size() != 0) {
            MouldGrouping mouldGrouping = mouldGroupings.get(0);
            id = mouldGrouping.getId();
        } else {
            //不存在的话就在当前选择的位置新建分组
            MouldGrouping mouldGrouping = new MouldGrouping();
            mouldGrouping.setMouldId(mouldId);
            mouldGrouping.setProjectId(projectId);
            mouldGrouping.setName("Schemas");
            mouldGroupingMapper.insert(mouldGrouping);
            //新建之后查询出分组，获取到分组的id
            id = mouldGroupingMapper.selectList(Wrappers.lambdaQuery(MouldGrouping.class)
                    .eq(MouldGrouping::getMouldId, mouldId)
                    .eq(MouldGrouping::getProjectId, projectId)
                    .eq(MouldGrouping::getName, "Schemas")).get(0).getId();
        }
        return id;
    }

    /**
     * 将未涉及到的数据模型都变成: 上次导入不存在
     *
     * @param longList 未涉及到的数据模型的主键id集合
     */
    public void updateStatusNo(List<Long> longList) {
        //未涉及到的数据模型都变成未同步
        if (longList.size() != 0) {
            for (Long id : longList) {
                MouldDoc mouldDoc = mouldDocMapper.selectById(id);
                mouldDoc.setStatus(SyncStatus.OTHER);
                mouldDocMapper.updateById(mouldDoc);
            }
        }
    }

    /**
     * 获取当前项目中所有的数据模型的主键id集合
     *
     * @param projectId 项目id
     * @return 当前项目中所有的数据模型的主键id集合
     */
    private List<Long> getIdList(Long projectId) {
        //当前项目中的所有数据模型
        List<MouldDoc> list = Optional.ofNullable(mouldDocMapper.selectList(Wrappers.lambdaQuery(MouldDoc.class)
                .eq(MouldDoc::getProjectId, projectId))).orElse(new ArrayList<>());
        //将主键id整理到集合中
        List<Long> longList = new ArrayList<>();
        if (list.size() != 0) {
            for (MouldDoc mouldDoc : list) {
                Long id = mouldDoc.getId();
                longList.add(id);
            }
        }
        return longList;
    }

}
