package org.matrix.local.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.matrix.local.entity.Project;
import org.matrix.local.entity.relation.UserProject;
import org.matrix.local.enums.Role;
import org.matrix.local.mapper.ProjectMapper;
import org.matrix.local.mapper.UserMapper;
import org.matrix.local.mapper.relation.UserProjectMapper;
import org.matrix.local.service.IProjectService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Optional;

import static java.lang.String.format;
import static java.util.Objects.nonNull;
import static org.matrix.local.enums.Role.ADMIN;
import static org.matrix.local.enums.Role.CREATOR;


/**
 * 项目相关Service
 */
@Slf4j
@Service
public class ProjectService extends ServiceImpl<ProjectMapper, Project> implements IProjectService {

    @Autowired
    private UserMapper userMapper;
    @Autowired
    private ProjectMapper projectMapper;
    @Autowired
    private UserProjectMapper userProjectMapper;

    /**
     * 新增项目
     * 新增数据不可附带id
     * 项目名唯一 不可重复
     *
     * @param project 项目信息对象
     * @return 保存的项目id
     */
    public Long create(Project project) {
        //检查是否附带了id
        Optional.of(project).map(Project::getId)
                .ifPresent(id -> {
                    throw new RuntimeException("新增数据不可附带id");
                });
        //检查同名项目是否已存在
        Optional.ofNullable(projectMapper.selectOne(Wrappers.<Project>lambdaQuery().eq(Project::getName, project.getName())))
                .ifPresent(id -> {
                    throw new RuntimeException(format("该项目名已存在 : %s", project.getName()));
                });
        //保存
        projectMapper.insert(project);
        log.info("[项目] 新增项目 : {}", project.getName());
        return project.getId();
    }

    /**
     * 修改项目
     * 根据id修改项目
     * 修改数据必须附带id且指定id的原数据必须存在
     * 项目名唯一 不可重复
     *
     * @param project 项目信息对象
     */
    public void update(Project project) {
        //检查id
        Long id = Optional.ofNullable(project).map(Project::getId)
                .orElseThrow(() -> new RuntimeException("修改数据必须指定id"));
        //检查原数据是否存在
        Project before = Optional.ofNullable(projectMapper.selectById(id))
                .orElseThrow(() -> new RuntimeException(format("id为 %s 的项目数据不存在", id)));
        //检查同名项目是否已存在
        Optional.of(projectMapper.selectOne(Wrappers.<Project>lambdaQuery().eq(Project::getName, project.getName())))
                .ifPresent(aId -> {
                    throw new RuntimeException(format("该项目名已存在 : %s", project.getName()));
                });
        //保存
        projectMapper.updateById(project);
        log.info("[项目] 更新项目 : {}  ——> {}", before.getName(), project.getName());
    }

    /**
     * 根据id删除项目
     *
     * @param id 指定id
     */
    public void delete(Long id) {
        //检查项目是否存在
        Project project = Optional.ofNullable(projectMapper.selectById(id))
                .orElseThrow(() -> new RuntimeException(format("id为 %s 的项目数据不存在", id)));
        //保存
        userMapper.deleteById(id);
        log.info("[项目] 删除项目 : {}", project.getName());
    }

    /**
     * 指定用户在项目中的角色
     * 把用户绑定到项目中
     * 如果绑定关系已存在 则更新其角色
     *
     * @param projectId 项目id
     * @param userId    用户id
     * @param role      用户在项目中的角色
     */
    public void bindUserRole(Long projectId, Long userId, String role) {
        if (Role.roles().contains(role)) {
            if (CREATOR.getName().equals(role) || ADMIN.getName().equals(role)) {
                throw new RuntimeException("该角色无法手动设置");
            }
            UserProject userProject = userProjectMapper.selectOne(Wrappers.<UserProject>lambdaQuery().eq(UserProject::getUserId, userId).eq(UserProject::getProjectId, projectId));
            if (nonNull(userProject)) {
                userProject.setRole(role);
                userProjectMapper.updateById(userProject);
            } else {
                userProjectMapper.insert(new UserProject()
                        .setUserId(userId)
                        .setProjectId(projectId)
                        .setRole(role));
            }
        } else {
            throw new RuntimeException("不符合约定的角色");
        }
    }

    /**
     * 把用户绑定为项目创建者
     *
     * @param projectId 项目id
     * @param userId    用户id
     */
    public void bindUserCreator(Long projectId, Long userId) {
        String role = CREATOR.getName();
        UserProject userProject = new UserProject()
                .setUserId(userId)
                .setProjectId(projectId)
                .setRole(role);
        userProjectMapper.insert(userProject);
    }

}
