package org.matrix.local.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.bson.Document;
import org.matrix.local.entity.Project;
import org.matrix.local.entity.User;
import org.matrix.local.entity.config.UserConfig;
import org.matrix.local.entity.relation.UserProject;
import org.matrix.local.entity.vo.LoginInfo;
import org.matrix.local.entity.vo.UserInfo;
import org.matrix.local.enums.ConfigType;
import org.matrix.local.mapper.ProjectMapper;
import org.matrix.local.mapper.UserMapper;
import org.matrix.local.mapper.relation.UserProjectMapper;
import org.matrix.local.service.IUserService;
import org.matrix.local.util.MD5Util;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;
import static java.util.Objects.nonNull;

/**
 * 用户相关Service
 */
@Slf4j
@Service
public class UserService extends ServiceImpl<UserMapper, User> implements IUserService {

    public static final String COLLECTION_NAME = "user_config";
    @Autowired
    private UserMapper userMapper;
    @Autowired
    private ProjectMapper projectMapper;
    @Autowired
    private UserProjectMapper userProjectMapper;
    @Autowired
    private MongoTemplate mongoTemplate;

    /**
     * 新增用户
     * 新增数据不可附带id
     * 账号名为唯一值 不可出现重复
     *
     * @param user 用户信息对象
     */
    public void create(User user) {
        if (nonNull(user)) {
            //检查是否附带了id
            Optional.of(user).map(User::getId)
                    .ifPresent(id -> {
                        throw new RuntimeException("新增数据不可附带id");
                    });
            //检查该账号是否已存在
            Optional.ofNullable(userMapper.selectOne(Wrappers.<User>lambdaQuery().eq(User::getAccount, user.getAccount())))
                    .ifPresent(id -> {
                        throw new RuntimeException(format("该账号已存在 : %s", user.getAccount()));
                    });
            //保存
            userMapper.insert(user);
            log.info("[用户] 新增用户 : {}", user.getAccount());
        }
    }

    /**
     * 修改用户
     * 根据id修改用户
     * 修改数据必须附带id且指定id的原数据必须存在
     * 只能修改名称和密码
     * 不可修改账号和id
     *
     * @param user 用户信息对象
     */
    public void update(User user) {
        //检查id
        Long id = Optional.ofNullable(user).map(User::getId)
                .orElseThrow(() -> new RuntimeException("修改数据必须指定id"));
        //检查原数据是否存在
        User before = Optional.ofNullable(userMapper.selectById(id))
                .orElseThrow(() -> new RuntimeException(format("id为 %s 的用户数据不存在", id)));
        //保存
        String account = before.getAccount();
        userMapper.updateById(user.setAccount(account));
        log.info("[用户] 更新用户 : {}", account);
    }

    /**
     * 根据id删除用户
     *
     * @param id 指定id
     */
    public void delete(Long id) {
        //检查账号是否存在
        User user = Optional.ofNullable(userMapper.selectById(id))
                .orElseThrow(() -> new RuntimeException(format("id为 %s 的用户数据不存在", id)));
        //保存
        userMapper.deleteById(id);
        log.info("[用户] 删除用户 : {}", user.getAccount());
    }

    /**
     * 如果是用户的第一次配置，即库里没有配置项，则全部保存
     * ● 如果已有配置项，分为部分更新和全部更新
     *
     * @param userConfig 用户配置
     * @param configType 配置类型
     * @return 是否更新/插入成功
     */
    @SneakyThrows
    public boolean upsertConfig(UserConfig userConfig, ConfigType configType) {
        Long userId = userConfig.getUserId();

        // 首先查询该用户是否存在,不存在就抛出异常
        User dbUser = userMapper.selectById(userId);
        if (Objects.isNull(dbUser)) {
            throw new RuntimeException(format("保存用户配置 -- 数据库里没有用户id = %d 的用户!", userId));
        }

        // 如果是用户的第一次配置，即配置库里没有配置项，则全部保存
        Query userQuery = Query.query(Criteria.where("userId").is(userId));
        boolean firstConfig = !mongoTemplate.exists(userQuery, "user_config");
        if (firstConfig) {
            mongoTemplate.save(userConfig, COLLECTION_NAME);
            return true;
        } else {
            Update update;

            if (configType == ConfigType.ALL) {
                log.info("[用户配置] 正在给id = {} 的用户更新<全局>配置,新的配置是 {}", userId, userConfig);
                Document doc = new Document();
                mongoTemplate.getConverter().write(userConfig, doc);
                update = Update.fromDocument(doc);
                mongoTemplate.upsert(userQuery, update, COLLECTION_NAME);
                return true;
            } else {
                // 通过反射获取userConfig的属性值
                String propertyName = configType.getPropertyName();
                Field field = userConfig.getClass().getDeclaredField(propertyName);
                field.setAccessible(true);
                Object value = field.get(userConfig);
                // 更新mongo中的数据
                log.info("[用户配置] 正在给id = {} 的用户更新<{}>配置,新的{}是 {}", userId, configType.getDes(), configType.getDes(), value);
                update = Update.update(configType.getPropertyName(), value);
                mongoTemplate.upsert(userQuery, update, COLLECTION_NAME);
                return true;
            }
        }
    }

    /**
     * 登录信息缓存Map
     * key为sessionId
     * value为对应的当前登录的user的id
     */
    @Getter
    private static Map<String, Long> sessionMap = new HashMap<>();

    /**
     * 用户登录
     * 用户登录成功后 缓存其对应的sessionId到Map中
     *
     * @param loginInfo 用户的账号、密码
     * @param sessionId 对应的sessionId
     * @return 登录是否成功
     */
    public boolean login(LoginInfo loginInfo, String sessionId) {
        String account = loginInfo.getAccount();
        String password = MD5Util.encode(loginInfo.getPassword());
        User user = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getAccount, account)
                .eq(User::getPassword, password));
        if (nonNull(user)) {
            Long userId = user.getId();
            sessionMap.put(sessionId, userId);
            return true;
        } else {
            return false;
        }
    }

    /**
     * 用户登录
     * 用户登录成功后 缓存其对应的sessionId到Map中
     * sessionId从上下文中取 无需输入
     *
     * @param loginInfo 用户的账号、密码
     * @return 登录是否成功
     */
    public boolean login(LoginInfo loginInfo) {
        String sessionId = RequestContextHolder.currentRequestAttributes().getSessionId();
        return login(loginInfo, sessionId);
    }

    /**
     * 查询当前登录的用户
     * 根据输入的sessionId查出其当前登录的user的信息
     *
     * @param sessionId sessionId
     * @return 当前登录的user的信息
     */
    public UserInfo findNow(String sessionId) {
        boolean hasLogin = sessionMap.containsKey(sessionId);
        if (hasLogin) {
            Long userId = sessionMap.get(sessionId);
            User user = userMapper.selectById(userId);
            if (nonNull(user)) {
                return userInfo(user);
            } else {
                throw new RuntimeException(format("未找到id为%s的用户", userId));
            }
        } else {
            throw new RuntimeException("当前没有已登录的用户");
        }
    }

    /**
     * 查询当前登录的用户
     * 根据sessionId查出其当前登录的user的信息
     * sessionId从上下文中取 无需输入
     *
     * @return 当前登录的user的信息
     */
    public UserInfo findNow() {
        String sessionId = RequestContextHolder.currentRequestAttributes().getSessionId();
        return findNow(sessionId);
    }

    /**
     * 查询当前登录的用户id
     * 根据sessionId查出其当前登录的user的id
     * sessionId从上下文中取 无需输入
     * 如果没有session信息则返回0
     * 该方法为静态方法 无需依赖注入
     *
     * @return 当前登录的user的信息
     */
    public static Long findNowUserId() {
        try {
            String sessionId = RequestContextHolder.currentRequestAttributes().getSessionId();
            return sessionMap.getOrDefault(sessionId, 0L);
        } catch (Exception e) {
            return 0L;
        }
    }

    /**
     * 登出用户
     * 输入sessionId 清除其在Map中的信息
     *
     * @param sessionId sessionId
     */
    public void logout(String sessionId) {
        sessionMap.remove(sessionId);
    }

    /**
     * 指定用户id
     * 再根据项目id查出当前登录用户在该项目中的角色
     *
     * @param userId    用户id
     * @param projectId 项目id
     * @return 角色
     */
    public String findRole(Long userId, Long projectId) {
        UserProject userProject = userProjectMapper.selectOne(Wrappers.<UserProject>lambdaQuery().eq(UserProject::getUserId, userId).eq(UserProject::getProjectId, projectId));
        return Optional.ofNullable(userProject)
                .map(UserProject::getRole)
                .orElse(Strings.EMPTY);
    }

    /**
     * 根据项目id查出当前登录用户在该项目中的角色
     *
     * @param projectId 项目id
     * @return 角色
     */
    public String findRole(Long projectId) {
        Long userId = findNow().getId();
        return findRole(userId, projectId);
    }

    /**
     * 判断用户是否是管理员账号
     *
     * @param userInfo 用户信息
     * @return bool结果
     */
    public Boolean isAdmin(UserInfo userInfo) {
        //TODO
        return false;
    }

    //-------------------------------private--------------------------------

    private UserInfo userInfo(User user) {
        List<Project> projects = userProjectMapper.selectList(Wrappers.<UserProject>lambdaQuery()
                .eq(UserProject::getUserId, user.getId())).stream()
                .map(UserProject::getProjectId)
                .map(projectMapper::selectById)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return new UserInfo(
                user.getId(),
                user.getName(),
                user.getAccount(),
                user.getPassword(),
                user.getRole(),
                projects
        );
    }

}
