package org.matrix.remote.service;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.extern.slf4j.Slf4j;
import org.matrix.datasource.IDataSourceService;
import org.matrix.local.entity.relation.UserProject;
import org.matrix.local.mapper.ProjectMapper;
import org.matrix.local.mapper.UserMapper;
import org.matrix.local.mapper.relation.UserProjectMapper;
import org.matrix.remote.entity.ZtProject;
import org.matrix.remote.entity.ZtUser;
import org.matrix.remote.entity.ZtUserView;
import org.matrix.remote.mapper.ChangeMapper;
import org.matrix.remote.mapper.ZtProjectMapper;
import org.matrix.remote.mapper.ZtUserMapper;
import org.matrix.remote.mapper.ZtUserViewMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 禅道数据全量同步工具
 */
@Slf4j
@Service
public class DataAllCollector {

    @Autowired
    private ChangeMapper changeMapper;
    @Autowired
    private ZtUserMapper ztUserMapper;
    @Autowired
    private ZtUserViewMapper ztUserViewMapper;
    @Autowired
    private ZtProjectMapper ztProjectMapper;
    @Autowired
    private UserMapper userMapper;
    @Autowired
    private ProjectMapper projectMapper;
    @Autowired
    private UserProjectMapper userProjectMapper;
    @Autowired
    private TransHelper transHelper;
    @Autowired
    private IDataSourceService dataSourceService;

    private List<ZtUser> ztUsers = new ArrayList<>();
    private List<ZtUserView> ztUserViews = new ArrayList<>();
    private List<ZtProject> ztProjects = new ArrayList<>();

    /**
     * 执行一次全量同步
     */
    public void execute() {
        log.info("全量同步禅道用户项目数据...");
        //清空增量记录
        changeMapper.delete(Wrappers.lambdaQuery());
        //用户数据
        collectUser();
        saveUser();
        //项目数据
        collectProject();
        saveProject();
        //用户项目关系数据
        collectUserView();
        saveUserProject();
        log.info("同步结束....");
    }

    //------------------------------------private-------------------------------------

    /**
     * 从禅道数据库读取用户数据
     */
    private void collectUser() {
        dataSourceService.switchDataSource("zentao");
        ztUsers.clear();
        ztUsers = ztUserMapper.selectList(Wrappers.lambdaQuery());
        dataSourceService.switchMainDataSource();
    }


    /**
     * 从禅道数据库读取项目数据
     */
    private void collectProject() {
        dataSourceService.switchDataSource("zentao");
        ztProjects.clear();
        ztProjects = ztProjectMapper.selectList(Wrappers.<ZtProject>lambdaQuery()
                .eq(ZtProject::getDeleted, "0")
                .eq(ZtProject::getProject, 0));
        dataSourceService.switchMainDataSource();
    }

    /**
     * 从禅道数据库读取用户相关项目关系数据
     */
    private void collectUserView() {
        dataSourceService.switchDataSource("zentao");
        ztUserViews.clear();
        ztUserViews = ztUserViewMapper.selectList(Wrappers.lambdaQuery());
        dataSourceService.switchMainDataSource();
    }

    /**
     * 将用户数据保存到本地
     */
    private void saveUser() {
        dataSourceService.switchMainDataSource();
        userMapper.delete(Wrappers.lambdaQuery());
        ztUsers.stream()
                .map(transHelper::user)
                .forEach(userMapper::insert);
    }

    /**
     * 将项目数据保存到本地
     */
    private void saveProject() {
        dataSourceService.switchMainDataSource();
        projectMapper.delete(Wrappers.lambdaQuery());
        ztProjects.stream()
                .map(transHelper::project)
                .forEach(projectMapper::insert);
    }

    /**
     * 将用户项目关系数据保存到本地
     */
    private void saveUserProject() {
        dataSourceService.switchDataSource("zentao");
        List<UserProject> userProjectList = ztUserViews.stream()
                .flatMap(ztUserView -> transHelper.userProjects(ztUserView).stream())
                .collect(Collectors.toList());
        dataSourceService.switchMainDataSource();
        userProjectMapper.delete(Wrappers.lambdaQuery());
        userProjectList.forEach(userProjectMapper::insert);
    }

}
