package org.matrix.remote.service;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import lombok.extern.slf4j.Slf4j;
import org.matrix.datasource.IDataSourceService;
import org.matrix.local.entity.Project;
import org.matrix.local.entity.User;
import org.matrix.local.entity.relation.UserProject;
import org.matrix.local.mapper.ProjectMapper;
import org.matrix.local.mapper.UserMapper;
import org.matrix.local.mapper.relation.UserProjectMapper;
import org.matrix.remote.entity.Change;
import org.matrix.remote.entity.ZtProject;
import org.matrix.remote.entity.ZtUser;
import org.matrix.remote.entity.ZtUserView;
import org.matrix.remote.mapper.ChangeMapper;
import org.matrix.remote.mapper.ZtProjectMapper;
import org.matrix.remote.mapper.ZtUserMapper;
import org.matrix.remote.mapper.ZtUserViewMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

import static java.util.Objects.nonNull;

/**
 * 禅道数据增量同步工具（基于触发器和专用表）
 */
@Slf4j
@Service
public class DataTriggerCollector {

    @Autowired
    private ChangeMapper changeMapper;
    @Autowired
    private ZtUserMapper ztUserMapper;
    @Autowired
    private ZtUserViewMapper ztUserViewMapper;
    @Autowired
    private ZtProjectMapper ztProjectMapper;
    @Autowired
    private UserMapper userMapper;
    @Autowired
    private ProjectMapper projectMapper;
    @Autowired
    private UserProjectMapper userProjectMapper;
    @Autowired
    private TransHelper transHelper;
    @Autowired
    private IDataSourceService dataSourceService;

    /**
     * 执行一次增量同步
     */
    public void execute() {
        dataSourceService.switchDataSource("zentao");
        Long count = changeMapper.selectCount(Wrappers.lambdaQuery());
        boolean hasChanges = count > 0;
        if (hasChanges) {
            List<Change> changes = changeMapper.selectList(Wrappers.lambdaQuery());
            changes.forEach(this::handle);
        }
        changeMapper.delete(Wrappers.lambdaQuery());
        dataSourceService.switchMainDataSource();
    }

    //----------------------------------------private-----------------------------------------

    /**
     * 增量数据同步处理
     * @param change 增量数据
     */
    private void handle(Change change) {
        String entity = change.getEntity();
        log.info("发现数据变更 数据类型:{} 操作类型:{} 主键:{}", change.getEntity(), change.getType(), change.getId());
        switch (entity) {
            case "user":
                handleUser(change);
                return;
            case "project":
                handleProject(change);
                return;
            case "userview":
                handleUserView(change);
                return;
            default:
        }
    }

    private void handleUser(Change change) {
        dataSourceService.switchDataSource("zentao");
        Integer id = Integer.valueOf(change.getId());
        ZtUser ztUser = ztUserMapper.selectById(id);
        dataSourceService.switchMainDataSource();
        String type = change.getType();
        switch (type) {
            case "insert":
                if (nonNull(ztUser)) {
                    User user = transHelper.user(ztUser);
                    userMapper.insert(user);
                }
                return;
            case "update":
                if (nonNull(ztUser)) {
                    User user = transHelper.user(ztUser);
                    userMapper.updateById(user);
                }
                return;
            case "delete":
                userMapper.deleteById(id);
                return;
            default:
        }
    }

    private void handleProject(Change change) {
        dataSourceService.switchDataSource("zentao");
        Integer id = Integer.valueOf(change.getId());
        ZtProject ztProject = ztProjectMapper.selectById(id);
        dataSourceService.switchMainDataSource();
        String type = change.getType();
        switch (type) {
            case "insert":
                if (nonNull(ztProject)) {
                    Project project = transHelper.project(ztProject);
                    projectMapper.insert(project);
                }
                return;
            case "update":
                if (nonNull(ztProject)) {
                    Project project = transHelper.project(ztProject);
                    projectMapper.updateById(project);
                }
                return;
            case "delete":
                projectMapper.deleteById(id);
                return;
            default:
        }
    }

    private void handleUserView(Change change) {
        dataSourceService.switchDataSource("zentao");
        String account = change.getId();
        ZtUserView ztUserView = ztUserViewMapper.selectOne(Wrappers.<ZtUserView>lambdaQuery().eq(ZtUserView::getAccount, account));
        dataSourceService.switchMainDataSource();
        String type = change.getType();
        switch (type) {
            case "insert":
            case "update":
                if (nonNull(ztUserView)) {
                    List<UserProject> userProjects = transHelper.userProjects(ztUserView);
                    userProjects.stream().findAny()
                            .map(UserProject::getUserId)
                            .ifPresent(userId -> userProjectMapper.delete(Wrappers.<UserProject>lambdaQuery().eq(UserProject::getUserId, userId)));
                    userProjects.forEach(userProjectMapper::insert);
                }
                return;
            case "delete":
                if (nonNull(ztUserView)) {
                    Long userId = userMapper.selectOne(Wrappers.<User>lambdaQuery().eq(User::getAccount, ztUserView.getAccount())).getId();
                    userProjectMapper.delete(Wrappers.<UserProject>lambdaQuery().eq(UserProject::getUserId, userId));
                }
                return;
            default:
        }
    }

}
