package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.entity.Case;
import org.matrix.exception.GlobalException;
import org.matrix.service.ICaseService;
import org.matrix.vo.InterfaceGroupVo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author mruny
 * @create 2022/8/2 10:39:01
 */
@CrossOrigin
@RestController
@RequestMapping("/cases")
@Api(tags = "对用例Case的基本操作")
public class CaseController {

    @Autowired
    private ICaseService caseService;

    @ApiOperation("根据主键id查询测试用例")
    @GetMapping("/id")
    public ResponseEntity<Case> findByIdCase(@RequestParam Long id) {
        Case aCase = Optional.ofNullable(caseService.getById(id)).orElse(new Case());
        return ResponseEntity.ok(aCase);
    }

    @ApiOperation("根据主键id删除测试用例")
    @DeleteMapping("/{id}")
    public ResponseEntity<String> removeCase(@PathVariable Long id) {
        caseService.removeById(id);
        return ResponseEntity.ok(String.format("删除成功，删除的接口文档id为: %d", id));
    }

    @ApiOperation("添加测试用例")
    @PostMapping
    public ResponseEntity<Case> insertCase(@RequestBody Case aCase) {
        return Optional.of(caseService.save(aCase)).orElseThrow(() -> new GlobalException("添加失败"))
                ? ResponseEntity.ok(aCase)
                : ResponseEntity.status(HttpStatus.BAD_REQUEST).body(aCase);
    }

    @ApiOperation("修改测试用例")
    @PutMapping
    public ResponseEntity<Case> updateCase(@RequestBody Case aCase) {
        return Optional.of(caseService.updateById(aCase)).orElseThrow(() -> new GlobalException("修改失败"))
                ? ResponseEntity.ok(aCase)
                : ResponseEntity.status(HttpStatus.BAD_REQUEST).body(aCase);
    }

    @ApiOperation("查询快捷请求")
    @GetMapping
    public ResponseEntity<List<Case>> findCase(@RequestParam Long projectId) {
        List<Case> cases = Optional.ofNullable(caseService.list(Wrappers.lambdaQuery(Case.class)
                .eq(Case::getProjectId, projectId)
                .eq(Case::getDocId, -1L))).orElse(new ArrayList<>());
        return ResponseEntity.ok(cases);
    }

    @ApiOperation("查询当前项目下所有分组,以及分组下的接口,测试用例等")
    @GetMapping("/all")
    public ResponseEntity<InterfaceGroupVo> findAll(@RequestParam Long projectId) {
        InterfaceGroupVo result = caseService.getAll(projectId);
        return ResponseEntity.ok(result);
    }

}
