package org.matrix.autotest.controller;

import com.google.common.collect.ImmutableMap;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.vo.CommonResult;
import org.matrix.vo.CommonResultObj;
import org.matrix.local.config.aop.Authorize;
import org.matrix.local.entity.Project;
import org.matrix.local.entity.vo.UserBindInfo;
import org.matrix.local.entity.vo.UserInfo;
import org.matrix.local.enums.Authority;
import org.matrix.local.service.impl.ProjectService;
import org.matrix.local.service.impl.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/projects")
@Api(tags = "对项目project的基本操作")
public class ProjectController {

    @Autowired
    private UserService userService;
    @Autowired
    private ProjectService projectService;

    @ApiOperation(value = "查询当前用户的相关项目")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Project>>> projects() {
        UserInfo userInfo = userService.findNow();
        return CommonResult.success(userInfo.getProjects(), "查询成功");
    }

    @ApiOperation(value = "新增项目", notes = "新增数据不可附带id;账号名为唯一值不可出现重复;当前用户将被设置为项目的创建者")
    @PostMapping
    public ResponseEntity<Map<String, String>> create(@RequestBody Project project) {
        Long projectId = projectService.create(project);
        UserInfo userInfo = userService.findNow();
        Long userId = userInfo.getId();
        projectService.bindUserCreator(projectId, userId);
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }

    @Authorize(Authority.MANAGE)
    @ApiOperation(value = "修改项目", notes = "根据id修改项目;修改数据必须附带id且指定id的原数据必须存在;项目名为唯一值不可出现重复;")
    @PutMapping
    public ResponseEntity<Map<String, String>> update(@RequestBody Project project) {
        projectService.update(project);
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }

    @Authorize(Authority.MANAGE)
    @ApiOperation(value = "保存项目绑定的用户角色", notes = "可多个;如果绑定关系已存在则更新其角色;")
    @PutMapping("/{projectId}")
    public ResponseEntity<Map<String, String>> saveUserRole(@PathVariable Long projectId, @RequestBody List<UserBindInfo> userBindInfos) {
        userBindInfos.forEach(userBindInfo -> projectService.bindUserRole(projectId, userBindInfo.getUserId(), userBindInfo.getRole()));
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }

    @Authorize(Authority.MANAGE)
    @ApiOperation(value = "删除项目", notes = "根据id删除项目")
    @DeleteMapping("/{id}")
    public ResponseEntity<Map<String, String>> delete(@PathVariable Long id) {
        projectService.delete(id);
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }


}
