package org.matrix.autotest.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.entity.Environment;
import org.matrix.exception.GlobalException;
import org.matrix.service.IEnvironmentService;
import org.matrix.service.IInterfaceDocService;
import org.matrix.service.IMouldDocService;
import org.matrix.vo.InterfaceVo;
import org.matrix.vo.MouldVo;
import org.matrix.vo.StatusCode;
import org.matrix.vo.swaggerEntityVo.AttributeInVo;
import org.matrix.vo.swaggerEntityVo.definitionsEntity.AttributeOutVo;
import org.matrix.vo.swaggerEntityVo.definitionsEntity.DefinitionsVo;
import org.matrix.vo.swaggerEntityVo.definitionsEntity.PropertiesVo;
import org.matrix.vo.swaggerEntityVo.responsesEntity.CodeStatusVo;
import org.matrix.vo.swaggerEntityVo.responsesEntity.ResponseVo;
import org.matrix.vo.swaggerEntityVo.swaggerEntity.PathVo;
import org.matrix.vo.swaggerEntityVo.swaggerEntity.SwaggerOuter;
import org.matrix.vo.swaggerEntityVo.swaggerEntity.TagSummary;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author mruny
 * @create 2022/6/27 10:43:26
 */
@CrossOrigin
@RestController
@RequestMapping("/swaggers")
@Api(tags = "swagger解析")
public class SwaggerHandleController {

    @Autowired
    private IMouldDocService mouldDocService;

    @Autowired
    private IInterfaceDocService interfaceDocService;

    @Autowired
    private IEnvironmentService environmentService;

    public static long i = 0;

    /**
     * 导入环境
     *
     * @param environment 环境对象
     * @return 数据导入情况
     */
    @ApiOperation("导入环境")
    @PostMapping("/environment")
    @Transactional(rollbackFor = Exception.class)
    public ResponseEntity<StatusCode> insertEnv(@RequestBody Environment environment) {
        StatusCode statusCode = environmentService.importEnv(environment);
        return ResponseEntity.ok(statusCode);
    }

    /**
     * 导入数据模型
     *
     * @param mouldVos  数据模型对象VO
     * @param projectId 项目id
     * @param pattern   导入类型
     * @param mouldId   上级分组
     * @return 数据导入情况
     */
    @ApiOperation("导入数据模型")
    @Transactional(rollbackFor = Exception.class)
    @PostMapping("/mouldDoc")
    public ResponseEntity<StatusCode> insertMouldDoc(@RequestBody List<MouldVo> mouldVos,
                                                     @RequestParam Long projectId,
                                                     @RequestParam int pattern,
                                                     @RequestParam Long mouldId) {
        StatusCode statusCode = mouldDocService.importMouldDoc(mouldVos, projectId, pattern, mouldId);
        return ResponseEntity.ok(statusCode);
    }

    /**
     * 导入接口文档
     *
     * @param interfaceVos 接口文档Vo类
     * @param projectId    项目id
     * @param pattern      导入方式
     * @param interfaceId  上级分组
     * @param envId        环境id
     * @return 导入结果
     */
    @ApiOperation("导入接口文档")
    @Transactional(rollbackFor = Exception.class)
    @PostMapping("/interfaceDoc")
    public ResponseEntity<StatusCode> insertInterfaceDoc(@RequestBody List<InterfaceVo> interfaceVos,
                                                         @RequestParam Long projectId,
                                                         @RequestParam int pattern,
                                                         @RequestParam Long interfaceId,
                                                         @RequestParam Long envId) {
        StatusCode statusCode = interfaceDocService.importInterfaceDoc(interfaceVos, projectId, envId, pattern, interfaceId);
        return ResponseEntity.ok(statusCode);
    }

    /**
     * 解析swaggerJson
     */
    @GetMapping
    @ApiOperation("获取并解析swagger数据")
    public SwaggerOuter handleSwagger(@RequestParam String url) {
        String loadJson = loadJson(url);
        SwaggerOuter json = new SwaggerOuter();
        if (loadJson != null && !"".equals(loadJson)) {
            JSONObject swaggerJson = JSONObject.parseObject(loadJson);
            String host = String.valueOf(swaggerJson.get("host"));
            json.setHost(host);
            String basePath = String.valueOf(swaggerJson.get("basePath"));
            json.setBasePath(basePath);
            List<DefinitionsVo> definitions = handleDefinitions(swaggerJson);
            json.setDefinitions(definitions);
            JSONObject paths = swaggerJson.getJSONObject("paths");
            List<PathVo> list = new ArrayList<>();
            if (paths != null) {
                for (Map.Entry<String, Object> stringObjectEntry : paths.entrySet()) {
                    String pathUrl = stringObjectEntry.getKey();
                    //请求方式
                    JSONObject pathJson = paths.getJSONObject(pathUrl);
                    Set<String> methodSets = pathJson.keySet();
                    if (!CollectionUtils.isEmpty(methodSets)) {
                        for (String httpMethod : methodSets) {
                            PathVo pathVo = new PathVo();
                            i = i + 1L;
                            pathVo.setIdentifierId(i);
                            JSONObject methodJson = pathJson.getJSONObject(httpMethod);
                            JSONArray tags = methodJson.getJSONArray("tags");
                            String tag = null;
                            if (!CollectionUtils.isEmpty(tags)) {
                                tag = tags.getString(0);
                            }
                            String summary = methodJson.getString("summary");
                            JSONArray parameters = methodJson.getJSONArray("parameters");
                            List<ResponseVo> responseVos = handleResponse(methodJson);
                            String operationId = methodJson.getString("operationId");
                            String deprecated = methodJson.getString("deprecated");
                            JSONArray consumes = methodJson.getJSONArray("consumes");
                            JSONArray produces = methodJson.getJSONArray("produces");
                            JSONArray security = methodJson.getJSONArray("security");
                            pathVo.setTags(tag);
                            pathVo.setHttpMethod(httpMethod);
                            pathVo.setPathUrl(pathUrl);
                            pathVo.setSummary(summary);
                            pathVo.setOperationId(operationId);
                            pathVo.setConsumes(consumes);
                            pathVo.setParameters(parameters);
                            pathVo.setResponses(responseVos);
                            pathVo.setDeprecated(deprecated);
                            pathVo.setSecurity(security);
                            pathVo.setProduces(produces);
                            list.add(pathVo);
                        }
                    }
                }
                List<TagSummary> tagSummaryList = new ArrayList<>();
                //stream取出key:tags,value:summary
                Map<String, List<PathVo>> collect = list.stream()
                        .filter(pathVo -> pathVo != null && pathVo.getTags() != null)
                        .collect(Collectors.groupingBy(PathVo::getTags));
                Set<String> tags = collect.keySet();
                for (String tag : tags) {
                    i = i + 1L;
                    TagSummary tagSummary = new TagSummary();
                    List<PathVo> pathVos = collect.get(tag);
                    tagSummary.setIdentifierId(i);
                    tagSummary.setTag(tag);
                    tagSummary.setPaths(pathVos);
                    tagSummaryList.add(tagSummary);
                }
                json.setTagSummaryList(tagSummaryList);
            }
        }
        return json;
    }

    /**
     * 读取swagger地址里的JSON信息
     *
     * @param url swagger地址
     * @return swagger中的JSON数据
     */
    public String loadJson(String url) {
        StringBuilder json = new StringBuilder();
        try {
            URL urlObject = new URL(url);
            URLConnection uc = urlObject.openConnection();
            InputStream inputStream = uc.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            String inputLine;
            while ((inputLine = reader.readLine()) != null) {
                json.append(inputLine);
            }
            reader.close();
        } catch (IOException e) {
            e.printStackTrace();
            throw new GlobalException(String.format("请求swagger数据失败,您读的swagger地址ip为 %s", url));
        }
        return json.toString();
    }

    /**
     * 处理responses中的数据
     */
    public List<ResponseVo> handleResponse(JSONObject methodJson) {
        List<ResponseVo> list = new ArrayList<>();
        JSONObject responses = methodJson.getJSONObject("responses");
        if (responses != null) {
            //所有状态码
            Set<String> codes = responses.keySet();
            for (String code : codes) {
                ResponseVo responseVo = new ResponseVo();
                JSONObject responseInside = responses.getJSONObject(code);
                responseVo.setCode(code);
                CodeStatusVo codeStatusVo = new CodeStatusVo();
                //获取描述
                String description = responseInside.getString("description");
                codeStatusVo.setDescription(description);
                if ("200".equals(code)) {
                    JSONObject schema = responseInside.getJSONObject("schema");
                    AttributeInVo attributeInVo = handleItems(schema);
                    codeStatusVo.setAttributeInVo(attributeInVo);
                } else {
                    //非200时状态码
                    responseVo.setCodeStatusVo(codeStatusVo);
                }
                responseVo.setCodeStatusVo(codeStatusVo);
                list.add(responseVo);
            }
        }
        return list;
    }

    /**
     * 解析host解析definitions
     */
    public List<DefinitionsVo> handleDefinitions(JSONObject swaggerJson) {
        //多个definitions,最外层
        List<DefinitionsVo> list = new ArrayList<>();
        //取到definitions内部的对象
        JSONObject definitions = swaggerJson.getJSONObject("definitions");
        if (definitions != null) {
            //对象名称集合
            Set<String> objNames = definitions.keySet();
            //获取到每个对象
            for (String objName : objNames) {
                i = i + 1L;
                //List内的最外层结构
                DefinitionsVo definitionsVo = new DefinitionsVo();
                definitionsVo.setIdentifierId(i);
                //definitionsVo的内部是attributeVo
                AttributeOutVo attributeOutVo = new AttributeOutVo();
                //根据对象名称可以取到，所有对象内部的信息，包括对象的类型type，properties，对象名称title，以及描述description
                JSONObject objInside = definitions.getJSONObject(objName);
                String type = objInside.getString("type");
                String title = objInside.getString("title");
                String description = objInside.getString("description");
                attributeOutVo.setType(type);
                attributeOutVo.setTitle(title);
                attributeOutVo.setDescription(description);
                //取到每个对象的properties
                JSONObject properties = objInside.getJSONObject("properties");
                if (properties != null) {
                    List<PropertiesVo> propertiesVoList = handleProperties(properties);
                    attributeOutVo.setPropertiesVoList(propertiesVoList);
                }
                //可能出现直接就是additionalProperties，
                JSONObject additionalProperties = objInside.getJSONObject("additionalProperties");
                if (additionalProperties != null) {
                    AttributeInVo attributeInVo = handleItems(additionalProperties);
                    attributeOutVo.setAdditionalProperties(attributeInVo);
                }
                definitionsVo.setObjName(objName);
                definitionsVo.setAttributeOutVo(attributeOutVo);
                list.add(definitionsVo);
            }
        }
        return list;
    }

    /**
     * 处理definitions内第一层的properties
     */
    public List<PropertiesVo> handleProperties(JSONObject properties) {
        List<PropertiesVo> list = new ArrayList<>();
        if (properties != null) {
            Set<String> attributeNames = properties.keySet();
            //每个对象中，可能存在多个属性
            for (String attributeName : attributeNames) {
                //获取到对象中属性的类型等信息
                JSONObject attributeInside = properties.getJSONObject(attributeName);
                //处理properties以及items
                AttributeInVo attributeInVo = handleItems(attributeInside);
                PropertiesVo propertiesVo = new PropertiesVo();
                propertiesVo.setName(attributeName);
                propertiesVo.setAttributeInVo(attributeInVo);
                list.add(propertiesVo);
            }
        }
        return list;
    }

    /**
     * 处理items
     */
    public AttributeInVo handleItems(JSONObject attributeInside) {
        AttributeInVo attributeInVo = new AttributeInVo();
        if (attributeInside != null) {
            //获取到属性的类型，根据不同的类型进行处理
            String type = attributeInside.getString("type");
            if (type != null && !"".equals(type)) {
                switch (type) {
                    case "string":
                        attributeInVo = handleString(attributeInside);
                        attributeInVo.setType(type);
                        break;
                    case "integer":
                        attributeInVo = handleInteger(attributeInside);
                        attributeInVo.setType(type);
                    case "object":
                        attributeInVo = handleObject(attributeInside);
                        attributeInVo.setType(type);
                        break;
                    case "array":
                        attributeInVo = handleArray(attributeInside);
                        attributeInVo.setType(type);
                        break;
                    case "boolean":
                        attributeInVo.setType(type);
                        break;
                    default:
                        break;
                }
            } else {
                String ref = attributeInside.getString("$ref");
                if (ref != null) {
                    attributeInVo.setRef(ref);
                }
            }
        }
        return attributeInVo;
    }

    /**
     * 处理String类型的情况
     */
    public AttributeInVo handleString(JSONObject attributeInside) {
        AttributeInVo attributeInVo = new AttributeInVo();
        JSONArray anEnum = attributeInside.getJSONArray("enum");
        String description = attributeInside.getString("description");
        attributeInVo.setDescription(description);
        if (anEnum != null) {
            List<String> enumTypes = anEnum.toJavaList(String.class);
            attributeInVo.setEnumType(enumTypes);
        }
        return attributeInVo;
    }

    /**
     * 处理integer类型的情况
     */
    public AttributeInVo handleInteger(JSONObject attributeInside) {
        AttributeInVo attributeInVo = new AttributeInVo();
        String description = attributeInside.getString("description");
        attributeInVo.setDescription(description);
        String format = attributeInside.getString("format");
        attributeInVo.setFormat(format);
        return attributeInVo;
    }

    /**
     * 处理array类型的情况
     */
    public AttributeInVo handleArray(JSONObject attributeInside) {
        AttributeInVo attributeInVo = new AttributeInVo();
        JSONArray example = attributeInside.getJSONArray("example");
        String description = attributeInside.getString("description");
        attributeInVo.setDescription(description);
        if (example != null) {
            List<String> javaList = example.toJavaList(String.class);
            attributeInVo.setExample(javaList);
        }
        JSONObject items = attributeInside.getJSONObject("items");
        if (items != null) {
            AttributeInVo child = handleItems(items);
            attributeInVo.setItems(child);
        }
        return attributeInVo;
    }

    /**
     * 处理object类型的情况
     */
    public AttributeInVo handleObject(JSONObject attributeInside) {
        AttributeInVo attributeInVo = new AttributeInVo();
        String description = attributeInside.getString("description");
        attributeInVo.setDescription(description);
        JSONObject additionalProperties = attributeInside.getJSONObject("additionalProperties");
        if (additionalProperties != null) {
            AttributeInVo child = handleItems(additionalProperties);
            attributeInVo.setItems(child);
        }
        return attributeInVo;
    }

}