package org.matrix.autotest.controller;

import com.google.common.collect.ImmutableMap;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.local.config.aop.Authorize;
import org.matrix.local.entity.User;
import org.matrix.local.entity.vo.LoginInfo;
import org.matrix.local.entity.vo.UserInfo;
import org.matrix.local.enums.Authority;
import org.matrix.local.enums.Role;
import org.matrix.local.service.impl.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

@Api(tags = "对用户user的基本操作")
@RestController
@RequestMapping("/user")
public class UserController {

    @Autowired
    private UserService userService;

    @Authorize(Authority.ADMIN)
    @ApiOperation(value = "新增用户", notes = "新增数据不可附带id;账号名为唯一值不可出现重复;")
    @PostMapping
    public ResponseEntity<Map<String, String>> create(@RequestBody User user) {
        userService.create(user);
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }

    @Authorize(Authority.ADMIN)
    @ApiOperation(value = "修改用户", notes = "根据id修改用户;修改数据必须附带id且指定id的原数据必须存在;账号名为唯一值不可出现重复;只能修改名称和密码而账号名不可修改;")
    @PutMapping
    public ResponseEntity<Map<String, String>> update(@RequestBody User user) {
        userService.update(user);
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }

    @Authorize(Authority.ADMIN)
    @ApiOperation(value = "删除用户", notes = "根据id删除用户")
    @DeleteMapping("/{id}")
    public ResponseEntity<Map<String, String>> delete(@PathVariable Long id) {
        userService.delete(id);
        return ResponseEntity.ok(ImmutableMap.of("message", "操作成功"));
    }

    @ApiOperation("用户登录")
    @PostMapping("/login")
    public ResponseEntity<Map<String, String>> login(@RequestBody LoginInfo loginInfo) {
        boolean success = userService.login(loginInfo);
        if (success) {
            return ResponseEntity.ok(ImmutableMap.of("message", "登录成功"));
        } else {
            return ResponseEntity.ok(ImmutableMap.of("message", "用户名或密码错误"));
        }
    }

    @ApiOperation("查询当前登录用户")
    @GetMapping("/login/info")
    public ResponseEntity<UserInfo> find() {
        UserInfo userInfo = userService.findNow();
        return ResponseEntity.ok(userInfo);
    }

    @ApiOperation("查看所有角色")
    @GetMapping("/roles")
    public ResponseEntity<Role[]> authorities() {
        return ResponseEntity.ok(Role.values());
    }

    @ApiOperation("用户登出")
    @GetMapping("/logout")
    public ResponseEntity<UserInfo> logout() {
        UserInfo userInfo = userService.findNow();
        return ResponseEntity.ok(userInfo);
    }

}
