package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.matrix.actuators.datasource.IDataSourceService;
import org.matrix.autotest.utils.PageTools;
import org.matrix.database.entity.Connect;
import org.matrix.database.service.IConnectService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.CachePut;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.sql.SQLException;
import java.util.Optional;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/connects")
@Api(tags = "对于数据库connect的基本操作")
public class ConnectController {

    private final IConnectService connectService;

    private final IDataSourceService dataSourceService;

    public ConnectController(IConnectService connectService, IDataSourceService dataSourceService) {
        this.connectService = connectService;
        this.dataSourceService = dataSourceService;
    }

    /**
     * 测试数据库连接
     *
     * @param connect 数据库连接参数对象
     * @return 返回连接信息, 如果失败会返回具体错误信息
     */
    @ApiOperation("测试数据库连接")
    @PostMapping("/test")
    public ResponseEntity<String> testConnect(@RequestBody Connect connect) {
        long startTime = System.currentTimeMillis();
        long endTime;
        try {
            dataSourceService.testConnection(connect.toDataSourceDTO());
        } catch (SQLException e) {
            endTime = System.currentTimeMillis();
            e.printStackTrace();
            log.warn("[数据库] 连接目标数据源数据库失败，目标数据源数据 = {}", connect);
            return ResponseEntity.status(503).body(String.format("连接失败,SQL状态 = %s 异常信息 = %s ,响应时间 %d ms", e.getSQLState(), e.getMessage(), endTime - startTime));
        }
        endTime = System.currentTimeMillis();
        return ResponseEntity.ok(String.format("连接成功!响应时间 %d ms", endTime - startTime));
    }

    /**
     * 分页查询所有数据库信息
     *
     * @param pageSize 每页多少条数据
     * @param pageNum  当前第几页
     * @return 分页查询的结果, 数据库信息
     */
    @ApiOperation(value = "分页查询数据库信息")
    @GetMapping("/page")
    @Cacheable(cacheNames = "connectPageCache",
            key = "#pageSize + '_' + #pageNum + '_' + #projectId",
            condition = "#pageNum != null && #pageSize != null",
            unless = "#result.statusCodeValue != 200")
    public ResponseEntity<CommonResultObj<Page<Connect>>> findPageConnects(
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(defaultValue = "1") int pageNum,
            @RequestParam Long projectId) {
        Page<Connect> results = Optional.ofNullable(connectService.page(Page.of(pageNum, pageSize), Wrappers.lambdaQuery(Connect.class)
                        .eq(Connect::getProjectId, projectId)))
                .orElse(new Page<>());
        PageTools.pageTool(pageSize, pageNum, results);
        return CommonResult.success(results, "查询成功");
    }

    /**
     * 条件查询数据库信息
     *
     * @param pageSize 每页多少条数据
     * @param pageNum  当前第几页
     * @param name     数据库名称
     * @return 分页查询的结果, 数据库信息
     */
    @ApiOperation(value = "条件查询数据库信息")
    @GetMapping("/condition")
    public ResponseEntity<CommonResultObj<Page<Connect>>> findConditionConnects(
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(defaultValue = "1") int pageNum,
            @RequestParam(required = false, defaultValue = "") String name,
            @RequestParam Long projectId) {
        Page<Connect> results = Optional.ofNullable(connectService.page(Page.of(pageNum, pageSize), Wrappers.lambdaQuery(Connect.class)
                        .eq(Connect::getProjectId, projectId)
                        .like(StringUtils.hasLength(name), Connect::getName, name)))
                .orElse(new Page<>());
        PageTools.pageTool(pageSize, pageNum, results);
        return CommonResult.success(results, "查询成功");
    }

    /**
     * 添加连接池
     *
     * @param connect 连接池对象
     * @return 添加的连接池, 以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加连接池")
    @PostMapping
    @Caching(
            put = {@CachePut(cacheNames = "connectPageCache", key = "#result.body.data.id",
                    condition = "#p0 != null", unless = "#result.statusCodeValue != 200")
            },
            evict = {@CacheEvict(cacheNames = "connectPageCache", allEntries = true)}
    )
    public ResponseEntity<CommonResultObj<Connect>> insertConnect(@RequestBody Connect connect) {
        return CommonResult.pred(connectService::save, connect, "添加成功", "添加失败");
    }

    /**
     * 修改连接池
     *
     * @param connect 连接池对象
     * @return 修改后的连接池, 以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改连接池")
    @PutMapping
    @Caching(
            put = {@CachePut(cacheNames = "connectPageCache", key = "#result.body.data.id",
                    condition = "#p0 != null", unless = "#result.statusCodeValue != 200")
            },
            evict = {@CacheEvict(cacheNames = "connectPageCache", allEntries = true)}
    )
    public ResponseEntity<CommonResultObj<Connect>> updateConnect(@RequestBody Connect connect) {
        return CommonResult.pred(connectService::updateById, connect, "修改成功", "修改失败");
    }

    /**
     * 根据主键id删除连接池
     *
     * @param id 连接池id
     * @return 删除的连接池id, 以及是否删除成功的提示信息
     */
    @ApiOperation(value = "根据主键id删除连接池")
    @DeleteMapping("/{id}")
    @CacheEvict(cacheNames = "connectPageCache", allEntries = true, condition = "#p0 != null")
    public ResponseEntity<CommonResultObj<Long>> deleteConnect(@PathVariable Long id) {
        return CommonResult.pred(connectService::removeById, id, "删除成功", "删除失败或id不存在");
    }

}
