package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.Project;
import org.matrix.autotest.service.ProjectService;
import org.matrix.autotest.utils.CommonResult;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@RestController
@RequestMapping("/projects")
@Api(tags = "对项目表project的基本操作")
public class ProjectController {

    private final ProjectService projectService;

    public ProjectController(ProjectService projectService) {
        this.projectService = projectService;
    }

    /**
     * 查询所有项目
     *
     * @return 查询到的所有项目
     */
    @ApiOperation(value = "查询所有项目")
    @GetMapping
    public CommonResult<List<Project>> findAllProject() {
        List<Project> results = projectService.findAllProject();
        if (results != null && results.size() != 0){
            return CommonResult.success(results);
        }else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询项目
     *
     * @param id 项目Id
     * @return 项目
     */
    @ApiOperation(value = "根据id查询项目")
    @GetMapping("/{id}")
    public CommonResult<Project> findByIdProject(@PathVariable Integer id) {
        Project result = projectService.findByIdProject(id);
        if (result != null){
            return CommonResult.success(result);
        }else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加项目
     *
     * @param project 项目
     * @return 添加的项目
     */
    @ApiOperation(value = "添加项目")
    @PostMapping
    public CommonResult<Project> insertProject(@RequestBody Project project) {
        int i = projectService.insertProject(project);
        if (i != 0){
            return CommonResult.success(project);
        }else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改项目
     *
     * @param project 项目
     * @return 修改后的项目
     */
    @ApiOperation(value = "根据id修改项目")
    @PutMapping("/{id}")
    public CommonResult<Project> updateProject(@RequestBody Project project) {
        int i = projectService.updateProject(project);
        if (i != 0){
            return CommonResult.success(project);
        }else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除项目
     *
     * @param id 项目id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除项目")
    @DeleteMapping("/{id}")
    public CommonResult<Project> deleteProject(@PathVariable Integer id) {
        int i = projectService.deleteProject(id);
        if (i != 0){
            return CommonResult.success("删除成功");
        }else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

}

