package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.Action;
import org.matrix.autotest.service.ActionService;
import org.matrix.autotest.utils.CommonResult;
import org.matrix.autotest.utils.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 动作 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/actions")
@Api(tags = "对动作表action的基本操作")
public class ActionController {

    private final ActionService actionService;

    public ActionController(ActionService actionService) {
        this.actionService = actionService;
    }

    /**
     * 查询所有动作
     *
     * @return 查询到的所有动作
     */
    @ApiOperation(value = "查询所有动作")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Action>>> findAllAction() {
        List<Action> results = actionService.findAllAction();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询动作
     *
     * @param id 动作Id
     * @return 动作
     */
    @ApiOperation(value = "根据id查询动作")
    @GetMapping("/{id}")
    public ResponseEntity<CommonResultObj<Action>> findByIdAction(@PathVariable Integer id) {
        Action result = actionService.findByIdAction(id);
        if (result != null) {
            return CommonResult.success(result, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加动作
     *
     * @param action 动作
     * @return 添加的动作
     */
    @ApiOperation(value = "添加动作")
    @PostMapping
    public ResponseEntity<CommonResultObj<Action>> insertAction(@RequestBody Action action) {
        int i = actionService.insertAction(action);
        if (i != 0) {
            return CommonResult.success(action, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改动作
     *
     * @param action 动作
     * @return 修改后的动作
     */
    @ApiOperation(value = "修改动作")
    @PutMapping
    public ResponseEntity<CommonResultObj<Action>> updateAction(@RequestBody Action action) {
        int i = actionService.updateAction(action);
        if (i != 0) {
            return CommonResult.success(action, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除动作
     *
     * @param id 动作id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除动作")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteAction(@PathVariable Integer id) {
        int i = actionService.deleteAction(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除", "删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<Action>>> findByProjectIdAction(@PathVariable Integer projectId) {
        List<Action> results = actionService.findByProjectIdAction(projectId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据行为id查询动作
     *
     * @param moveId 行为id
     * @return 符合行为id的动作
     */
    @ApiOperation(value = "根据行为id查询所有")
    @GetMapping("/move/{moveId}")
    public ResponseEntity<CommonResultObj<List<Action>>> findByMoveIdAction(@PathVariable Integer moveId) {
        List<Action> results = actionService.findByMoveIdAction(moveId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

}

