package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.Move;
import org.matrix.autotest.service.MoveService;
import org.matrix.autotest.utils.CommonResult;
import org.matrix.autotest.utils.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/moves")
@Api(tags = "对行为表move的基本操作")
public class MoveController {

    private final MoveService moveService;

    public MoveController(MoveService moveService) {
        this.moveService = moveService;
    }

    /**
     * 查询所有行为
     *
     * @return 查询到的所有行为
     */
    @ApiOperation(value = "查询所有行为")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Move>>> findAllMove() {
        List<Move> results = moveService.findAllMove();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询行为
     *
     * @param id 行为Id
     * @return 行为
     */
    @ApiOperation(value = "根据id查询行为")
    @GetMapping("/{id}")
    public ResponseEntity<CommonResultObj<Move>> findByIdMove(@PathVariable Integer id) {
        Move result = moveService.findByIdMove(id);
        if (result != null) {
            return CommonResult.success(result, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加行为
     *
     * @param move 行为
     * @return 添加的行为
     */
    @ApiOperation(value = "添加行为")
    @PostMapping
    public ResponseEntity<CommonResultObj<Move>> insertMove(@RequestBody Move move) {
        int i = moveService.insertMove(move);
        if (i != 0) {
            return CommonResult.success(move, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改行为
     *
     * @param move 行为
     * @return 修改后的行为
     */
    @ApiOperation(value = "修改行为")
    @PutMapping
    public ResponseEntity<CommonResultObj<Move>> updateMove(@RequestBody Move move) {
        int i = moveService.updateMove(move);
        if (i != 0) {
            return CommonResult.success(move, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除行为
     *
     * @param id 行为id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除行为")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteMove(@PathVariable Integer id) {
        int i = moveService.deleteMove(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除", "删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<Move>>> findByProjectIdMove(@PathVariable Integer projectId) {
        List<Move> results = moveService.findByProjectIdMove(projectId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

}

