package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.TestCase;
import org.matrix.autotest.service.TestCaseService;
import org.matrix.autotest.utils.CommonResult;
import org.matrix.autotest.utils.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/testCases")
@Api(tags = "对用例表test_case的基本操作")
public class TestCaseController {

    private final TestCaseService testCaseService;

    public TestCaseController(TestCaseService testCaseService) {
        this.testCaseService = testCaseService;
    }

    /**
     * 查询所有用例
     *
     * @return 查询到的所有用例
     */
    @ApiOperation(value = "查询所有用例")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<TestCase>>> findAllTestCase() {
        List<TestCase> results = testCaseService.findAllTestCase();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询用例
     *
     * @param id 用例Id
     * @return 用例
     */
    @ApiOperation(value = "根据id查询用例")
    @GetMapping("/{id}")
    public ResponseEntity<CommonResultObj<TestCase>> findByIdTestCase(@PathVariable Integer id) {
        TestCase result = testCaseService.findByIdTestCase(id);
        if (result != null) {
            return CommonResult.success(result, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加用例
     *
     * @param testCase 用例
     * @return 添加的用例
     */
    @ApiOperation(value = "添加用例")
    @PostMapping
    public ResponseEntity<CommonResultObj<TestCase>> insertTestCase(@RequestBody TestCase testCase) {
        int i = testCaseService.insertTestCase(testCase);
        if (i != 0) {
            return CommonResult.success(testCase, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改用例
     *
     * @param testCase 用例
     * @return 修改后的用例
     */
    @ApiOperation(value = "修改用例")
    @PutMapping
    public ResponseEntity<CommonResultObj<TestCase>> updateTestCase(@RequestBody TestCase testCase) {
        int i = testCaseService.updateTestCase(testCase);
        if (i != 0) {
            return CommonResult.success(testCase, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除用例
     *
     * @param id 用例id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除用例")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteTestCase(@PathVariable Integer id) {
        int i = testCaseService.deleteTestCase(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除", "删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<TestCase>>> findByProjectIdTestCase(@PathVariable Integer projectId) {
        List<TestCase> results = testCaseService.findByProjectIdTestCase(projectId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

}

