package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.utils.PageTools;
import org.matrix.database.entity.DynamicVariable;
import org.matrix.database.service.IDynamicVariableService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.CachePut;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.Optional;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/dynamicVariables")
@Api(tags = "对动参dynamic_variable的基本操作")
public class DynamicVariableController {

    private final IDynamicVariableService dynamicVariableService;

    public DynamicVariableController(IDynamicVariableService dynamicVariableService) {
        this.dynamicVariableService = dynamicVariableService;
    }

    /**
     * 分页查询所有动参
     *
     * @param pageSize  每页多少条数据
     * @param pageNum   当前第几页
     * @param name      动参名称
     * @param projectId 项目id
     * @return 分页查询的结果, 动参
     */
    @ApiOperation(value = "分页查询动参")
    @GetMapping("/{projectId}")
    @Cacheable(cacheNames = "dynamicVariablePageCache",
            key = "#pageSize + '_' + #pageNum + '_' + #projectId",
            condition = "#pageNum != null && #pageSize != null",
            unless = "#result.statusCodeValue != 200")
    public ResponseEntity<CommonResultObj<Page<DynamicVariable>>> findPageDynamicVariable(
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(defaultValue = "1") int pageNum,
            @RequestParam(required = false, defaultValue = "") String name,
            @PathVariable Long projectId) {
        Page<DynamicVariable> results = Optional.ofNullable(dynamicVariableService.page(Page.of(pageNum, pageSize)
                , Wrappers.lambdaQuery(DynamicVariable.class).eq(DynamicVariable::getProjectId, projectId)
                        .like(StringUtils.hasLength(name)
                                , DynamicVariable::getName, name))).orElse(new Page<>());
        PageTools.pageTool(pageSize, pageNum, results);
        return CommonResult.success(results, "查询成功");
    }

    /**
     * 条件查询动参
     *
     * @param pageSize  每页多少条数据
     * @param pageNum   当前第几页
     * @param name      动参名称
     * @param projectId 项目id
     * @return 分页查询的结果, 动参
     */
    @ApiOperation(value = "条件查询动参")
    @GetMapping("/condition/{projectId}")
    public ResponseEntity<CommonResultObj<Page<DynamicVariable>>> findConditionDynamicVariable(
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(defaultValue = "1") int pageNum,
            @RequestParam(required = false, defaultValue = "") String name,
            @PathVariable Long projectId) {
        Page<DynamicVariable> results = Optional.ofNullable(dynamicVariableService.page(Page.of(pageNum, pageSize)
                , Wrappers.lambdaQuery(DynamicVariable.class).eq(DynamicVariable::getProjectId, projectId)
                        .like(StringUtils.hasLength(name)
                                , DynamicVariable::getName, name))).orElse(new Page<>());
        PageTools.pageTool(pageSize, pageNum, results);
        return CommonResult.success(results, "查询成功");
    }

    /**
     * 添加动态变量
     *
     * @param dynamicVariable 动态变量对象
     * @return 添加的动态变量, 以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加动态变量")
    @PostMapping
    @Caching(
            put = {@CachePut(cacheNames = "dynamicVariablePageCache", key = "#result.body.data.id",
                    condition = "#p0 != null", unless = "#result.statusCodeValue != 200")
            },
            evict = {@CacheEvict(cacheNames = "dynamicVariablePageCache", allEntries = true)}
    )
    public ResponseEntity<CommonResultObj<DynamicVariable>> insertDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        return CommonResult.pred(dynamicVariableService::save, dynamicVariable
                , "添加成功", "添加失败");
    }

    /**
     * 修改动态变量
     *
     * @param dynamicVariable 动态变量对象
     * @return 修改后的动态变量, 以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改动态变量")
    @PutMapping
    @Caching(
            put = {@CachePut(cacheNames = "dynamicVariablePageCache", key = "#result.body.data.id",
                    condition = "#p0 != null", unless = "#result.statusCodeValue != 200")
            },
            evict = {@CacheEvict(cacheNames = "dynamicVariablePageCache", key = "#p0.id",
                    condition = "#p0 != null")}
    )
    public ResponseEntity<CommonResultObj<DynamicVariable>> updateDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        return CommonResult.pred(dynamicVariableService::updateById, dynamicVariable
                , "修改成功", "修改失败");
    }

    /**
     * 删除动态变量
     *
     * @param id 动态变量id
     * @return 删除的动态变量id, 以及是否删除成功的提示信息
     */
    @ApiOperation(value = "根据主键id删除动态变量")
    @DeleteMapping("/{id}")
    @CacheEvict(cacheNames = "dynamicVariablePageCache", allEntries = true, condition = "#p0 != null")
    public ResponseEntity<CommonResultObj<Long>> deleteDynamicVariable(@PathVariable Long id) {
        return CommonResult.pred(dynamicVariableService::removeById, id
                , "删除成功", "删除失败或id不存在");
    }

}
