package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.Connect;
import org.matrix.database.service.IConnectService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.matrix.database.vo.PageResult;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/connects")
@Api(tags = "对连接池connect的基本操作")
public class ConnectController {

    private final IConnectService connectService;

    public ConnectController(IConnectService connectService) {
        this.connectService = connectService;
    }

    @ApiOperation(value = "分页查询")
    @GetMapping("/page")
    public ResponseEntity<CommonResultObj<PageResult>> findAllPage(PageResult pageResult) {
        pageResult = connectService.pageAll(pageResult);
        if (pageResult.getRows() != null) {
            return CommonResult.success(pageResult, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    @ApiOperation(value = "查询所有")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Connect>>> findAll() {
        List<Connect> list = connectService.list();
        if (list != null && list.size() != 0) {
            return CommonResult.success(list, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加连接池
     *
     * @param connect 连接池
     * @return 添加的连接池
     */
    @ApiOperation(value = "添加连接池")
    @PostMapping
    public ResponseEntity<CommonResultObj<Connect>> insertConnect(@RequestBody Connect connect) {
        int i = connectService.insertConnect(connect);
        if (i != 0) {
            return CommonResult.success(connect, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改连接池
     *
     * @param connect 连接池
     * @return 修改后的连接池
     */
    @ApiOperation(value = "修改连接池")
    @PutMapping
    public ResponseEntity<CommonResultObj<Connect>> updateConnect(@RequestBody Connect connect) {
        int i = connectService.updateConnect(connect);
        if (i != 0) {
            return CommonResult.success(connect, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除连接池
     *
     * @param id 连接池id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除连接池")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteConnect(@PathVariable Integer id) {
        int i = connectService.deleteConnect(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除", "删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

}

