package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.Action;
import org.matrix.database.entity.Move;
import org.matrix.database.service.IActionService;
import org.matrix.database.service.IMoveService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.matrix.database.vo.MoveAction;
import org.matrix.database.vo.PageResult;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/moves")
@Api(tags = "对行为move的基本操作")
public class MoveController {

    private final IActionService actionService;

    private final IMoveService moveService;

    public MoveController(IMoveService moveService, IActionService actionService) {
        this.moveService = moveService;
        this.actionService = actionService;
    }

    @ApiOperation(value = "分页查询用例")
    @GetMapping("/page")
    public ResponseEntity<CommonResultObj<PageResult>> findAllPage(PageResult pageResult) {
        pageResult = moveService.pageAll(pageResult);
        if (pageResult.getRows() != null) {
            return CommonResult.success(pageResult, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    @ApiOperation(value = "根据项目id查,行为以及,行为下的动作")
    @GetMapping("/{id}")
    public ResponseEntity<CommonResultObj<MoveAction>> findById(@PathVariable Integer id) {
        Move move = moveService.getById(id);
        Integer projectId = move.getProjectId();
        QueryWrapper<Action> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("project_id", projectId);
        List<Action> list = actionService.list(queryWrapper);
        MoveAction moveAction = new MoveAction();
        moveAction.setMove(move);
        moveAction.setAction(list);
        if (list != null && list.size() != 0) {
            return CommonResult.success(moveAction, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    @ApiOperation(value = "根据项目id查行为")
    @GetMapping("project/{projectId}")
    public ResponseEntity<CommonResultObj<List<Move>>> findByProjectIdMove(@PathVariable Integer projectId) {
        QueryWrapper<Move> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("project_id", projectId);
        List<Move> list = moveService.list(queryWrapper);
        if (list.size() != 0) {
            return CommonResult.success(list, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    @ApiOperation(value = "添加行为和动作")
    @PostMapping
    public ResponseEntity<CommonResultObj<MoveAction>> insertMoveAction(@RequestBody MoveAction moveAction) {
        boolean save = moveService.save(moveAction.getMove());
        Integer id = Math.toIntExact(moveAction.getMove().getId());
        for (Action action : moveAction.getAction()) {
            action.setMoveId(id);
        }
        boolean saveBatch = actionService.saveBatch(moveAction.getAction());
        if (save && saveBatch) {
            return CommonResult.success(moveAction, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    @ApiOperation(value = "修改行为和动作")
    @PutMapping
    public ResponseEntity<CommonResultObj<MoveAction>> updateMoveAction(@RequestBody MoveAction moveAction) {
        boolean moveUpdate = moveService.update(null);
        boolean actionUpdate = actionService.update(null);
        if (moveUpdate || actionUpdate) {
            return CommonResult.success(moveAction, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    @ApiOperation(value = "删除行为和动作")
    @DeleteMapping("/{projectId}")
    public ResponseEntity<CommonResultObj<MoveAction>> deleteMoveAction(@PathVariable Integer projectId) {
        QueryWrapper<Action> actionQueryWrapper = new QueryWrapper<>();
        actionQueryWrapper.eq("project_id", projectId);
        boolean action = actionService.remove(actionQueryWrapper);
        QueryWrapper<Move> moveQueryWrapper = new QueryWrapper<>();
        moveQueryWrapper.eq("project_id", projectId);
        boolean move = moveService.remove(moveQueryWrapper);
        if (action && move) {
            return CommonResult.success("删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

}

