package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.DynamicVariable;
import org.matrix.database.service.IDynamicVariableService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.matrix.database.vo.PageResult;
import org.matrix.exception.GlobalException;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Optional;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/dynamicVariables")
@Api(tags = "对动参dynamic_variable的基本操作")
public class DynamicVariableController {

    private final IDynamicVariableService dynamicVariableService;

    public DynamicVariableController(IDynamicVariableService dynamicVariableService) {
        this.dynamicVariableService = dynamicVariableService;
    }

    @ApiOperation(value = "分页查询")
    @PostMapping("/page")
    public ResponseEntity<CommonResultObj<PageResult>> findAllPage(@RequestBody PageResult pageResult) {
        pageResult = dynamicVariableService.pageAll(pageResult);
        if (pageResult.getRows() != null) {
            return CommonResult.success(pageResult, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }


    @ApiOperation(value = "查询所有动参")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<DynamicVariable>>> findAll() {
        List<DynamicVariable> list = dynamicVariableService.list();
        if (list != null && list.size() != 0) {
            return CommonResult.success(list, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加动态变量
     *
     * @param dynamicVariable 动态变量
     * @return 添加的动态变量
     */
    @ApiOperation(value = "添加动态变量")
    @PostMapping
    public ResponseEntity<CommonResultObj<DynamicVariable>> insertDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        int i = dynamicVariableService.insertDynamicVariable(dynamicVariable);
        if (i != 0) {
            return CommonResult.success(dynamicVariable, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改动态变量
     *
     * @param dynamicVariable 动态变量
     * @return 修改后的动态变量
     */
    @ApiOperation(value = "修改动态变量")
    @PutMapping
    public ResponseEntity<CommonResultObj<DynamicVariable>> updateDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        int i = dynamicVariableService.updateDynamicVariable(dynamicVariable);
        if (i != 0) {
            return CommonResult.success(dynamicVariable, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除动态变量
     *
     * @param id 动态变量id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除动态变量")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteDynamicVariable(@PathVariable Integer id) {
        int i = dynamicVariableService.deleteDynamicVariable(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除", "删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<DynamicVariable>>> findByProjectIdDynamicVariable(@PathVariable Long projectId) {
        List<DynamicVariable> results = Optional.ofNullable(dynamicVariableService.list(Wrappers.lambdaQuery(DynamicVariable.class)
                        .eq(DynamicVariable::getProjectId, projectId)))
                .orElseThrow(() -> new GlobalException(String.format("没有查到指定的行为,你提供的行为id是%d", projectId)));
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

}