package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.TestCase;
import org.matrix.database.service.ITestCaseService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.matrix.database.vo.PageResult;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/testCases")
@Api(tags = "对用例test_case的基本操作")
public class TestCaseController {

    private final ITestCaseService testCaseService;

    public TestCaseController(ITestCaseService testCaseService) {
        this.testCaseService = testCaseService;
    }

    @ApiOperation(value = "分页查询用例")
    @PostMapping("/page")
    public ResponseEntity<CommonResultObj<PageResult>> findAllPage(@RequestBody PageResult pageResult) {
        pageResult = testCaseService.pageAll(pageResult);
        if (pageResult.getRows() != null) {
            return CommonResult.success(pageResult, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    @ApiOperation(value = "查询所有用例")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<TestCase>>> findAll() {
        List<TestCase> list = testCaseService.list();
        if (list != null && list.size() != 0) {
            return CommonResult.success(list, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加用例
     *
     * @param testCase 用例
     * @return 添加的用例
     */
    @ApiOperation(value = "添加用例")
    @PostMapping
    public ResponseEntity<CommonResultObj<TestCase>> insertTestCase(@RequestBody TestCase testCase) {
        int i = testCaseService.insertTestCase(testCase);
        if (i != 0) {
            return CommonResult.success(testCase, "添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改用例
     *
     * @param testCase 用例
     * @return 修改后的用例
     */
    @ApiOperation(value = "修改用例")
    @PutMapping
    public ResponseEntity<CommonResultObj<TestCase>> updateTestCase(@RequestBody TestCase testCase) {
        int i = testCaseService.updateTestCase(testCase);
        if (i != 0) {
            return CommonResult.success(testCase, "修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除用例
     *
     * @param id 用例id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除用例")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteTestCase(@PathVariable Integer id) {
        int i = testCaseService.deleteTestCase(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除", "删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<TestCase>>> findByProjectIdTestCase(@PathVariable Integer projectId) {
        List<TestCase> results = testCaseService.findByProjectIdTestCase(projectId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results, "查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

}

