package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.Action;
import org.matrix.autotest.service.ActionService;
import org.matrix.autotest.utils.CommonResult;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 动作 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@RestController
@RequestMapping("/actions")
@Api(tags = "对动作表action的基本操作")
public class ActionController {

    private final ActionService actionService;

    public ActionController(ActionService actionService) {
        this.actionService = actionService;
    }

    /**
     * 查询所有动作
     *
     * @return 查询到的所有动作
     */
    @ApiOperation(value = "查询所有动作")
    @GetMapping
    public CommonResult<List<Action>> findAllAction() {
        List<Action> results = actionService.findAllAction();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results);
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询动作
     *
     * @param id 动作Id
     * @return 动作
     */
    @ApiOperation(value = "根据id查询动作")
    @GetMapping("/{id}")
    public CommonResult<Action> findByIdAction(@PathVariable Integer id) {
        Action result = actionService.findByIdAction(id);
        if (result != null) {
            return CommonResult.success(result);
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加动作
     *
     * @param action 动作
     * @return 添加的动作
     */
    @ApiOperation(value = "添加动作")
    @PostMapping
    public CommonResult<Action> insertAction(@RequestBody Action action) {
        int i = actionService.insertAction(action);
        if (i != 0) {
            return CommonResult.success(action);
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改动作
     *
     * @param action 动作
     * @return 修改后的动作
     */
    @ApiOperation(value = "根据id修改动作")
    @PutMapping("/{id}")
    public CommonResult<Action> updateAction(@RequestBody Action action) {
        int i = actionService.updateAction(action);
        if (i != 0) {
            return CommonResult.success(action);
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除动作
     *
     * @param id 动作id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除动作")
    @DeleteMapping("/{id}")
    public CommonResult<Action> deleteAction(@PathVariable Integer id) {
        int i = actionService.deleteAction(id);
        if (i != 0) {
            return CommonResult.success("删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/{projectId}")
    public CommonResult<List<Action>> findByProjectIdAction(@PathVariable Integer projectId) {
        List<Action> results = actionService.findByProjectIdAction(projectId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results);
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
        //todo 未完成
    }


}

