package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.DynamicVariable;
import org.matrix.autotest.service.DynamicVariableService;
import org.matrix.autotest.utils.CommonResult;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@RestController
@RequestMapping("/dynamicVariables")
@Api(tags = "对动态变量表dynamic_variable的基本操作")
public class DynamicVariableController {

    private final DynamicVariableService dynamicVariableService;

    public DynamicVariableController(DynamicVariableService dynamicVariableService) {
        this.dynamicVariableService = dynamicVariableService;
    }

    /**
     * 查询所有动态变量
     *
     * @return 查询到的所有动态变量
     */
    @ApiOperation(value = "查询所有动态变量")
    @GetMapping
    public CommonResult<List<DynamicVariable>> findAllDynamicVariable() {
        List<DynamicVariable> results = dynamicVariableService.findAllDynamicVariable();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results);
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询动态变量
     *
     * @param id 动态变量Id
     * @return 动态变量
     */
    @ApiOperation(value = "根据id查询动态变量")
    @GetMapping("/{id}")
    public CommonResult<DynamicVariable> findByIdDynamicVariable(@PathVariable Integer id) {
        DynamicVariable result = dynamicVariableService.findByIdDynamicVariable(id);
        if (result != null) {
            return CommonResult.success(result);
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加动态变量
     *
     * @param dynamicVariable 动态变量
     * @return 添加的动态变量
     */
    @ApiOperation(value = "添加动态变量")
    @PostMapping
    public CommonResult<DynamicVariable> insertDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        int i = dynamicVariableService.insertDynamicVariable(dynamicVariable);
        if (i != 0) {
            return CommonResult.success(dynamicVariable);
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改动态变量
     *
     * @param dynamicVariable 动态变量
     * @return 修改后的动态变量
     */
    @ApiOperation(value = "根据id修改动态变量")
    @PutMapping("/{id}")
    public CommonResult<DynamicVariable> updateDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        int i = dynamicVariableService.updateDynamicVariable(dynamicVariable);
        if (i != 0) {
            return CommonResult.success(dynamicVariable);
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除动态变量
     *
     * @param id 动态变量id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除动态变量")
    @DeleteMapping("/{id}")
    public CommonResult<DynamicVariable> deleteDynamicVariable(@PathVariable Integer id) {
        int i = dynamicVariableService.deleteDynamicVariable(id);
        if (i != 0) {
            return CommonResult.success("删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

}