package org.matrix.util;

import io.netty.handler.codec.http.HttpHeaderValues;
import org.apache.commons.lang3.StringUtils;
import org.matrix.config.HttpRequestConfig;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.client.CookieStore;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.matrix.entity.httpRequest.*;
import org.springframework.http.HttpMethod;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * HttpClient调用封装
 *
 * @author HuangXiahao
 * @version V1.0
 * @class AuthenticationUtils
 * @packageName com.example.personnelmanager.common.utils
 **/
public class HttpClientUtil {

    HttpRequestConfig config;

    CookieStore cookieStore;

    CloseableHttpClient client;


    public HttpClientUtil(HttpRequestConfig config) throws KeyStoreException, NoSuchAlgorithmException, KeyManagementException {
        this.config = config;
        this.cookieStore = config.cookieStore();
        this.client = config.client();
    }

    public CloseableHttpClient getClient() {
        return client;
    }

    /**
     * 初始化请求内数据并补全动态变量
     * @param httpRequestDetail
     */
    public void execute(HttpRequestDetail httpRequestDetail){
        for (RequestHeader header : httpRequestDetail.getHeaders()) {
            //获取动态变量，并将动态变量替换进去
            //todo
//            header.setValue();
        }
        for (RequestBody requestBody : httpRequestDetail.getRequestBodies()) {
            //todo

        }
        if (StringUtils.isEmpty(httpRequestDetail.getStringValue())){
            //todo

        }
            //todo
//        httpRequestDetail.setUrl();
    }


    /**
     * 发起Http请求
     *
     * @param httpRequestDetail
     * @return
     */
    private HttpResponseDetail sendHttpRequest(HttpRequestDetail httpRequestDetail) {
        CloseableHttpResponse response = null;
        Date startTime = new Date();
        String url = initUrlString(httpRequestDetail);
        HttpRequestBase requestBase = initHttpRequestBase(url, httpRequestDetail.getMethod());
        if (requestBase instanceof HttpEntityEnclosingRequestBase) {
            HttpEntity httpRequestBase = initHttpEntity(httpRequestDetail);
            ((HttpEntityEnclosingRequestBase) requestBase).setEntity(httpRequestBase);
        }
        requestBase.setHeaders(httpRequestDetail.getHeadersArray());
        try {
            response = getClient().execute(requestBase);
            Date endTime = new Date();
            return new HttpResponseDetail(
                    response,
                    EntityUtils.toString(response.getEntity(), "UTF-8"),
                    response.getStatusLine().getStatusCode(),
                    endTime.getTime() - startTime.getTime()
            );
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            //关闭请求request
            closeRequest(requestBase);
            //关闭response
            closeResponse(response);
        }
    }

    /**
     * 初始化请求HttpEntity
     *
     * @param httpRequestDetail
     * @return
     */
    public HttpEntity initHttpEntity(HttpRequestDetail httpRequestDetail) {
        HttpEntity httpEntity = null;
        switch (httpRequestDetail.getRequestType()) {
            case FORM_DATA:
                httpEntity = getMultipartEntity(httpRequestDetail.getRequestBodies());
                break;
            case X_WWW_FORM_URLENCODED:
                httpEntity = getUrlEncodeFormEntity(httpRequestDetail.getRequestBodies());
                break;
            case JSON:
                httpEntity =  getStringEntity(httpRequestDetail.getStringValue());
                httpRequestDetail.addHeaders(HttpHeaders.CONTENT_TYPE, HttpHeaderValues.APPLICATION_JSON.toString());
                break;
            case XML:
                httpEntity = getStringEntity(httpRequestDetail.getStringValue());
                httpRequestDetail.addHeaders(HttpHeaders.CONTENT_TYPE, HttpHeaderValues.APPLICATION_XML.toString());
                break;
            case TEXT:
                httpEntity = getStringEntity(httpRequestDetail.getStringValue());
                break;
            case BINARY:
                httpEntity = getByteArrayRequestEntity(httpRequestDetail.getStringValue());
                break;
            default:
                httpEntity = getEmptyStringEntity();
        }
        return httpEntity;
    }

    public HttpEntity getMultipartEntity(List<RequestBody> requestBodies) {
        MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
        for (RequestBody requestBody : requestBodies) {
            switch (requestBody.getType()) {
                case TEXT:
                    multipartEntityBuilder.addTextBody(requestBody.getName(), requestBody.getValue(), ContentType.create("text/plain", Consts.UTF_8));
                    break;
                case FILE:
                    multipartEntityBuilder.addBinaryBody(requestBody.getName(), getFileByPath(requestBody.getValue()));
                    break;
            }
        }
        return multipartEntityBuilder.build();
    }

    public HttpEntity getUrlEncodeFormEntity(List<RequestBody> requestBodies) {
        List<NameValuePair> param = new ArrayList<>();
        for (RequestBody requestBody : requestBodies) {
            param.add(new BasicNameValuePair(requestBody.getName(), requestBody.getValue()));
        }
        return new UrlEncodedFormEntity(param, StandardCharsets.UTF_8);
    }


    public StringEntity getEmptyStringEntity() {
        return new StringEntity(new String(), StandardCharsets.UTF_8);
    }

    public StringEntity getStringEntity(String value) {
        return new StringEntity(value, StandardCharsets.UTF_8);
    }


    public File getFileByPath(String path) {
        return new File(path);
    }

    public HttpEntity getByteArrayRequestEntity(String path) {
        try {
            File file = getFileByPath(path);
            byte[] bytesArray = new byte[(int) file.length()];
            FileInputStream fis = new FileInputStream(file);
            fis.read(bytesArray); // read file into bytes[]
            fis.close();
            return new ByteArrayEntity(bytesArray);
        } catch (IOException ioException) {
            ioException.printStackTrace();
            return getEmptyStringEntity();
        }


    }

    public String initUrlString(HttpRequestDetail httpRequestDetail) {
        String url = httpRequestDetail.getUrl();
        try {
            switch (httpRequestDetail.getRequestType()) {
                case REST:
                    String regex = "(?<=\\{)(.*?)(?=\\})";
                    Pattern pattern = Pattern.compile(regex);
                    Matcher mat = pattern.matcher(url);
                    while (mat.find()) {
                        url = url.replaceAll("\\{" + mat.group() + "\\}", "1");
                    }
                    break;
                case QUERY:
                    if (httpRequestDetail.getMethod().equals(HttpMethod.GET)) {
                        URIBuilder uriBuilder = null;
                        uriBuilder = new URIBuilder(url);
                        for (RequestBody requestBody : httpRequestDetail.getRequestBodies()) {
                            switch (requestBody.getType()) {
                                case TEXT:
                                    uriBuilder.setParameter(requestBody.getName(), requestBody.getValue());
                                    break;
                                case FILE:
                                    throw new NullPointerException("QUERY请求不能传入文件流");
                            }
                        }
                        url = uriBuilder.build().toString();
                    }
                    break;
            }
        } catch (URISyntaxException e) {
            e.printStackTrace();
        }
        return url;
    }

    /**
     * 初始化HttpRequestBase
     *
     * @param url
     * @param httpMethod
     * @return
     */
    public HttpRequestBase initHttpRequestBase(String url, HttpMethod httpMethod) {
        HttpRequestBase requestBase;
        switch (httpMethod) {
            case GET:
                requestBase = new HttpGet(url);
                break;
            case POST:
                requestBase = new HttpPost(url);
                break;
            case PUT:
                requestBase = new HttpPut(url);
                break;
            case DELETE:
                requestBase = new HttpDelete(url);
                break;
            case PATCH:
                requestBase = new HttpPatch(url);
                break;
            default:
                throw new IllegalStateException("不支持该类型的HTTP请求: " + httpMethod);
        }
        return requestBase;
    }

    public static void main(String[] args) throws MalformedURLException, URISyntaxException {

//        JSONObject jsonObject = new JSONObject();
//        jsonObject.put("method","POST");
//        Vo vo = JSON.parseObject(jsonObject.toJSONString(), Vo.class);
//        System.out.println("1");

//        String regex = "\\{([^\\}]+)\\}";

        HttpRequestDetail httpRequestDetail = new HttpRequestDetail();
        httpRequestDetail.getRequestBodies().add(new RequestBody("name", MultiPartRequestBodyType.TEXT, "张三"));
        httpRequestDetail.getRequestBodies().add(new RequestBody("text", MultiPartRequestBodyType.TEXT, "a1"));

        httpRequestDetail.setMethod(HttpMethod.GET);
        httpRequestDetail.setUrl("www.baidu.com");
        httpRequestDetail.setRequestType(HttpRequestType.QUERY);
    }

    /**
     * 关闭response（注意：用完记得关，不然会一直占用系统资源）
     *
     * @param response
     */
    public void closeResponse(CloseableHttpResponse response) {
        if (response != null) {
            if (response.getEntity() != null) {
                EntityUtils.consumeQuietly(response.getEntity());
            }
            try {
                response.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 关闭requestBase（注意：用完记得关，不然会一直占用系统资源）
     *
     * @param requestBase
     */
    public void closeRequest(HttpRequestBase requestBase) {
        requestBase.abort();
        requestBase.releaseConnection();
    }


}
