package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.TestJob;
import org.matrix.database.service.ITestJobService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.matrix.database.vo.TestJobVo;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.CachePut;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author mry
 */
@CrossOrigin
@RestController
@RequestMapping("/testJob")
@Api(tags = "对测试任务test_job的基本操作")
public class TestJobController {

    private final ITestJobService testJobService;

    public TestJobController(ITestJobService testJobService) {
        this.testJobService = testJobService;
    }

    /**
     * 查询所有测试任务
     *
     * @return 查询到的测试任务，以及是否查询成功
     */
    @ApiOperation(value = "查询所有测试任务")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<TestJob>>> findConnects() {
        List<TestJob> results = Optional.ofNullable(testJobService.list()).orElse(new ArrayList<>());
        return CommonResult.success(results, "查询成功");
    }

    /**
     * 条件查询所有测试任务信息
     *
     * @param pageSize  每页多少条数据
     * @param pageNum   当前第几页
     * @param projectId 项目id
     * @return 条件查询的结果，测试任务信息
     */
    @ApiOperation(value = "条件查询测试任务")
    @GetMapping("/page")
    @Cacheable(cacheNames = "testJobPageCache",
            key = "#pageSize + '_' + #pageNum + '_' + #projectId",
            condition = "#pageSize != null && #pageNum != null",
            unless = "#result.statusCodeValue != 200")
    public ResponseEntity<CommonResultObj<IPage<TestJobVo>>> findPageTestTasks(
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(defaultValue = "1") int pageNum,
            @RequestParam Long projectId
    ) {
        IPage<TestJobVo> result = testJobService.pageTestJob(projectId, "", pageSize, pageNum);
        return CommonResult.success(result, "查询成功");
    }

    /**
     * 条件查询所有测试任务信息
     *
     * @param pageSize  每页多少条数据
     * @param pageNum   当前第几页
     * @param projectId 项目id
     * @param name      测试任务名称
     * @return 条件查询的结果，测试任务信息
     */
    @ApiOperation(value = "条件查询测试任务")
    @GetMapping("/condition")
    public ResponseEntity<CommonResultObj<IPage<TestJobVo>>> findConditionTestTasks(
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(defaultValue = "1") int pageNum,
            @RequestParam Long projectId,
            @RequestParam(required = false, defaultValue = "") String name
    ) {
        IPage<TestJobVo> result = testJobService.pageTestJob(projectId, name, pageSize, pageNum);
        return CommonResult.success(result, "查询成功");
    }

    /**
     * 添加测试任务
     *
     * @param testJob 测试任务对象
     * @return 添加的测试任务，以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加测试任务")
    @PostMapping
    @Caching(
            put = {@CachePut(cacheNames = "testJobPageCache", key = "#result.body.data.id",
                    condition = "#p0 != null", unless = "#result.statusCodeValue != 200")
            },
            evict = {@CacheEvict(cacheNames = "testJobPageCache", allEntries = true)}
    )
    public ResponseEntity<CommonResultObj<TestJob>> insertTestTask(@RequestBody TestJob testJob) {
        return CommonResult.pred(testJobService::save, testJob
                , "添加成功", "添加失败");
    }

    /**
     * 修改测试任务
     *
     * @param testJob 测试任务对象
     * @return 修改后的测试任务，以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改测试任务")
    @PutMapping
    @Caching(
            put = {@CachePut(cacheNames = "testJobPageCache", key = "#result.body.data.id",
                    condition = "#p0 != null", unless = "#result.statusCodeValue != 200")
            },
            evict = {@CacheEvict(cacheNames = "testJobPageCache", allEntries = true)}
    )
    public ResponseEntity<CommonResultObj<TestJob>> updateTestTask(@RequestBody TestJob testJob) {
        return CommonResult.pred(testJobService::updateById, testJob
                , "修改成功", "修改失败");
    }

    /**
     * 根据主键id删除测试任务
     *
     * @param id 主键id
     * @return 删除的测试任务id，以及是否删除成功的提示
     */
    @ApiOperation(value = "根据主键id删除测试任务")
    @DeleteMapping("/{id}")
    @CacheEvict(cacheNames = "testJobPageCache", allEntries = true, condition = "#p0 != null")
    public ResponseEntity<CommonResultObj<Long>> deleteTestTask(@PathVariable Long id) {
        return CommonResult.pred(testJobService::removeById, id
                , "删除成功", "删除失败或id不存在");
    }
}
