package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.Connect;
import org.matrix.autotest.service.ConnectService;
import org.matrix.autotest.utils.CommonResult;
import org.matrix.autotest.utils.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/connects")
@Api(tags = "对连接池表connect的基本操作")
public class ConnectController {

    private final ConnectService connectService;

    public ConnectController(ConnectService connectService) {
        this.connectService = connectService;
    }

    /**
     * 查询所有连接池
     *
     * @return 查询到的所有连接池
     */
    @ApiOperation(value = "查询所有连接池")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Connect>>> findAllConnect() {
        List<Connect> results = connectService.findAllConnect();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results,"查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询连接池
     *
     * @param id 连接池Id
     * @return 连接池
     */
    @ApiOperation(value = "根据id查询连接池")
    @GetMapping("/{id}")
    public ResponseEntity<CommonResultObj<Connect>> findByIdConnect(@PathVariable Integer id) {
        Connect result = connectService.findByIdConnect(id);
        if (result != null) {
            return CommonResult.success(result,"查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加连接池
     *
     * @param connect 连接池
     * @return 添加的连接池
     */
    @ApiOperation(value = "添加连接池")
    @PostMapping
    public ResponseEntity<CommonResultObj<Connect>> insertConnect(@RequestBody Connect connect) {
        int i = connectService.insertConnect(connect);
        if (i != 0) {
            return CommonResult.success(connect,"添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改连接池
     *
     * @param connect 连接池
     * @return 修改后的连接池
     */
    @ApiOperation(value = "根据id修改连接池")
    @PutMapping("/{id}")
    public ResponseEntity<CommonResultObj<Connect>> updateConnect(@RequestBody Connect connect) {
        int i = connectService.updateConnect(connect);
        if (i != 0) {
            return CommonResult.success(connect,"修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除连接池
     *
     * @param id 连接池id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除连接池")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteConnect(@PathVariable Integer id) {
        int i = connectService.deleteConnect(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除","删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

}

