package org.matrix.autotest.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.autotest.entity.Environment;
import org.matrix.autotest.service.EnvironmentService;
import org.matrix.autotest.utils.CommonResult;
import org.matrix.autotest.utils.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * <p>
 * 实例表，项目对应的环境实例，例如：实验室环境，开发环境等 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/environments")
@Api(tags = "对实例表environment的基本操作")
public class EnvironmentController {

    private final EnvironmentService environmentService;

    public EnvironmentController(EnvironmentService environmentService) {
        this.environmentService = environmentService;
    }

    /**
     * 查询所有实例
     *
     * @return 查询到的所有实例
     */
    @ApiOperation(value = "查询所有实例")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Environment>>> findAllEnvironment() {
        List<Environment> results = environmentService.findAllEnvironment();
        if (results != null && results.size() != 0) {
            return CommonResult.success(results,"查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 根据id查询实例
     *
     * @param id 实例Id
     * @return 实例
     */
    @ApiOperation(value = "根据id查询实例")
    @GetMapping("/{id}")
    public ResponseEntity<CommonResultObj<Environment>> findByIdEnvironment(@PathVariable Integer id) {
        Environment result = environmentService.findByIdEnvironment(id);
        if (result != null) {
            return CommonResult.success(result,"查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

    /**
     * 添加实例
     *
     * @param environment 实例
     * @return 添加的实例
     */
    @ApiOperation(value = "添加实例")
    @PostMapping
    public ResponseEntity<CommonResultObj<Environment>> insertEnvironment(@RequestBody Environment environment) {
        int i = environmentService.insertEnvironment(environment);
        if (i != 0) {
            return CommonResult.success(environment,"添加成功");
        } else {
            return CommonResult.failed("添加失败");
        }
    }

    /**
     * 修改实例
     *
     * @param environment 实例
     * @return 修改后的实例
     */
    @ApiOperation(value = "根据id修改实例")
    @PutMapping("/{id}")
    public ResponseEntity<CommonResultObj<Environment>> updateEnvironment(@RequestBody Environment environment) {
        int i = environmentService.updateEnvironment(environment);
        if (i != 0) {
            return CommonResult.success(environment,"修改成功");
        } else {
            return CommonResult.failed("修改失败");
        }
    }

    /**
     * 删除实例
     *
     * @param id 实例id
     * @return 是否删除成功
     */
    @ApiOperation(value = "根据id删除实例")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<String>> deleteEnvironment(@PathVariable Integer id) {
        int i = environmentService.deleteEnvironment(id);
        if (i != 0) {
            return CommonResult.success("id: " + id + "已删除","删除成功");
        } else {
            return CommonResult.failed("删除失败,或不存在需要删除的数据");
        }
    }

    /**
     * 根据项目id查询动作
     *
     * @param projectId 项目id
     * @return 符合项目id的动作
     */
    @ApiOperation(value = "根据项目id查询所有")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<Environment>>> findByProjectIdEnvironment(@PathVariable Integer projectId) {
        List<Environment> results = environmentService.findByProjectIdEnvironment(projectId);
        if (results != null && results.size() != 0) {
            return CommonResult.success(results,"查询成功");
        } else {
            return CommonResult.failed("查询失败或无数据");
        }
    }

}

