package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.TestTask;
import org.matrix.database.service.ITestTaskService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author mry
 */
@CrossOrigin
@RestController
@RequestMapping("/testTasks")
@Api(tags = "对测试任务test_task的基本操作")
public class TestTaskController {

    private final ITestTaskService testTaskService;

    public TestTaskController(ITestTaskService testTaskService) {
        this.testTaskService = testTaskService;
    }

    /**
     * 查询所有测试任务
     *
     * @return 查询到的测试任务，以及是否查询成功
     */
    @ApiOperation(value = "查询所有测试任务")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<TestTask>>> findConnects() {
        List<TestTask> results = Optional.ofNullable(testTaskService.list()).orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

    /**
     * 分页查询所有测试任务信息
     *
     * @param pageSize 每页多少条数据
     * @param pageNum  当前第几页
     * @param name     测试任务名称
     * @return 分页查询的结果，测试任务信息
     */
    @ApiOperation(value = "分页查询测试任务")
    @GetMapping("/page")
    public ResponseEntity<CommonResultObj<Page<TestTask>>> findPageTestTasks(
            int pageSize,
            int pageNum,
            String name) {
        Page<TestTask> results = Optional.of(testTaskService.page(Page.of(pageNum, pageSize), Wrappers.lambdaQuery(TestTask.class)
                        .like(StringUtils.hasLength(name), TestTask::getName, name)))
                .orElse(new Page<>());
        return results.getRecords().size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

    /**
     * 添加测试任务
     *
     * @param testTask 测试任务对象
     * @return 添加的测试任务，以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加测试任务")
    @PostMapping
    public ResponseEntity<CommonResultObj<TestTask>> insertTestTask(@RequestBody TestTask testTask) {
        return CommonResult.pred(testTaskService::save, testTask
                , "添加成功", "添加失败");
    }

    /**
     * 修改测试任务
     *
     * @param testTask 测试任务对象
     * @return 修改后的测试任务，以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改测试任务")
    @PutMapping
    public ResponseEntity<CommonResultObj<TestTask>> updateTestTask(@RequestBody TestTask testTask) {
        return CommonResult.pred(testTaskService::updateById, testTask
                , "修改成功", "修改失败");
    }

    /**
     * 根据主键id删除测试任务
     *
     * @param id 主键id
     * @return 删除的测试任务id，以及是否删除成功的提示
     */
    @ApiOperation(value = "根据主键id删除测试任务")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<Long>> deleteTestTask(@PathVariable Long id) {
        return CommonResult.pred(testTaskService::removeById, id
                , "删除成功", "删除失败或id不存在");
    }
}
