package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.DynamicVariable;
import org.matrix.database.service.IDynamicVariableService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/dynamicVariables")
@Api(tags = "对动参dynamic_variable的基本操作")
public class DynamicVariableController {

    private final IDynamicVariableService dynamicVariableService;

    public DynamicVariableController(IDynamicVariableService dynamicVariableService) {
        this.dynamicVariableService = dynamicVariableService;
    }

    /**
     * 查询所有动态参数
     *
     * @return 查询到的所有动态参数, 以及是否查询成功的提示信息
     */
    @ApiOperation(value = "查询所有动态变量")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<DynamicVariable>>> findAllDynamicVariable() {
        List<DynamicVariable> results = Optional.ofNullable(dynamicVariableService.list()).orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

    /**
     * 添加动态变量
     *
     * @param dynamicVariable 动态变量对象
     * @return 添加的动态变量, 以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加动态变量")
    @PostMapping
    public ResponseEntity<CommonResultObj<DynamicVariable>> insertDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        return CommonResult.pred(dynamicVariableService::save, dynamicVariable
                , "添加成功", "添加失败");
    }

    /**
     * 修改动态变量
     *
     * @param dynamicVariable 动态变量对象
     * @return 修改后的动态变量, 以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改动态变量")
    @PutMapping
    public ResponseEntity<CommonResultObj<DynamicVariable>> updateDynamicVariable(@RequestBody DynamicVariable dynamicVariable) {
        return CommonResult.pred(dynamicVariableService::updateById, dynamicVariable
                , "修改成功", "修改失败");
    }

    /**
     * 删除动态变量
     *
     * @param id 动态变量id
     * @return 删除的动态变量id, 以及是否删除成功的提示信息
     */
    @ApiOperation(value = "根据主键id删除动态变量")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<Long>> deleteDynamicVariable(@PathVariable Long id) {
        return CommonResult.pred(dynamicVariableService::removeById, id
                , "删除成功", "删除失败或id不存在");
    }

    /**
     * 根据项目id查询动态变量
     *
     * @param projectId 项目id
     * @return 符合项目id的动态变量, 以及是否查询成功的提示信息
     */
    @ApiOperation(value = "根据项目id查询所有动态变量")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<DynamicVariable>>> findByProjectIdDynamicVariable(@PathVariable Long projectId) {
        List<DynamicVariable> results = Optional.ofNullable(dynamicVariableService.list(Wrappers.lambdaQuery(DynamicVariable.class)
                        .eq(DynamicVariable::getProjectId, projectId)))
                .orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

}
