package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.Environment;
import org.matrix.database.service.IEnvironmentService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <p>
 * 实例表，项目对应的环境实例，例如：实验室环境，开发环境等 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/environments")
@Api(tags = "对环境environment的基本操作")
public class EnvironmentController {

    private final IEnvironmentService environmentService;

    public EnvironmentController(IEnvironmentService environmentService) {
        this.environmentService = environmentService;
    }

    /**
     * 查询所有环境
     *
     * @return 查询到的所有环境, 以及是否查询成功的提示信息
     */
    @ApiOperation(value = "查询所有环境")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<Environment>>> findAll() {
        List<Environment> results = Optional.ofNullable(environmentService.list()).orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

    /**
     * 添加环境
     *
     * @param environment 环境对象
     * @return 添加的环境, 以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加环境")
    @PostMapping
    public ResponseEntity<CommonResultObj<Environment>> insertEnvironment(@RequestBody Environment environment) {
        return CommonResult.pred(environmentService::save, environment
                , "添加成功", "添加失败");
    }

    /**
     * 修改环境
     *
     * @param environment 环境对象
     * @return 修改后的环境, 以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改环境")
    @PutMapping
    public ResponseEntity<CommonResultObj<Environment>> updateEnvironment(@RequestBody Environment environment) {
        return CommonResult.pred(environmentService::updateById, environment
                , "修改成功", "修改失败");
    }

    /**
     * 删除环境
     *
     * @param id 环境id
     * @return 删除的环境id, 以及是否删除成功的提示信息
     */
    @ApiOperation(value = "根据主键id删除环境")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<Long>> deleteEnvironment(@PathVariable Long id) {
        return CommonResult.pred(environmentService::removeById, id
                , "删除成功", "删除失败或id不存在");
    }

    /**
     * 根据项目id查询环境
     *
     * @param projectId 项目id
     * @return 符合项目id的环境, 以及是否查询成功的提示信息
     */
    @ApiOperation(value = "根据项目id查询所有环境")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<Environment>>> findByProjectIdEnvironment(@PathVariable Long projectId) {
        List<Environment> results = Optional.ofNullable(environmentService.list(Wrappers.lambdaQuery(Environment.class)
                        .eq(Environment::getProjectId, projectId)))
                .orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

}
