package org.matrix.autotest.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.matrix.database.entity.TestCase;
import org.matrix.database.service.ITestCaseService;
import org.matrix.database.vo.CommonResult;
import org.matrix.database.vo.CommonResultObj;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <p>
 * 前端控制器
 * </p>
 *
 * @author mry
 * @since 2022-01-07
 */
@CrossOrigin
@RestController
@RequestMapping("/testCases")
@Api(tags = "对用例test_case的基本操作")
public class TestCaseController {

    private final ITestCaseService testCaseService;

    public TestCaseController(ITestCaseService testCaseService) {
        this.testCaseService = testCaseService;
    }

    /**
     * 查询所有用例
     *
     * @return 查询到的所有用例, 以及是否查询成功的提示信息
     */
    @ApiOperation(value = "查询所有用例")
    @GetMapping
    public ResponseEntity<CommonResultObj<List<TestCase>>> findAll() {
        List<TestCase> results = Optional.ofNullable(testCaseService.list()).orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

    /**
     * 添加用例
     *
     * @param testCase 用例对象
     * @return 添加的用例, 以及是否添加成功的提示信息
     */
    @ApiOperation(value = "添加用例")
    @PostMapping
    public ResponseEntity<CommonResultObj<TestCase>> insertTestCase(@RequestBody TestCase testCase) {
        return CommonResult.pred(testCaseService::save, testCase
                , "添加成功", "添加失败");
    }

    /**
     * 修改用例
     *
     * @param testCase 用例对象
     * @return 修改后的用例, 以及是否修改成功的提示信息
     */
    @ApiOperation(value = "修改用例")
    @PutMapping
    public ResponseEntity<CommonResultObj<TestCase>> updateTestCase(@RequestBody TestCase testCase) {
        return CommonResult.pred(testCaseService::updateById, testCase
                , "修改成功", "修改失败");
    }

    /**
     * 删除用例
     *
     * @param id 用例id
     * @return 删除的用例id, 以及是否删除成功的提示信息
     */
    @ApiOperation(value = "根据主键id删除用例")
    @DeleteMapping("/{id}")
    public ResponseEntity<CommonResultObj<Long>> deleteTestCase(@PathVariable Long id) {
        return CommonResult.pred(testCaseService::removeById, id
                , "删除成功", "删除失败或id不存在");
    }

    /**
     * 根据项目id查询用例
     *
     * @param projectId 项目id
     * @return 符合项目id的环境, 以及是否查询成功的提示信息
     */
    @ApiOperation(value = "根据项目id查询所有用例")
    @GetMapping("/project/{projectId}")
    public ResponseEntity<CommonResultObj<List<TestCase>>> findByProjectIdTestCase(@PathVariable Integer projectId) {
        List<TestCase> results = Optional.ofNullable(testCaseService.list(Wrappers.lambdaQuery(TestCase.class)
                        .eq(TestCase::getProjectId, projectId)))
                .orElse(new ArrayList<>());
        return results.size() != 0
                ? CommonResult.success(results, "查询成功")
                : CommonResult.failed(results, "查询失败或无数据");
    }

}
