package com.matrix.md.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.matrix.md.entity.Component;
import com.matrix.md.entity.Specification;
import com.matrix.md.entity.vo.InterVo;
import com.matrix.md.entity.vo.SpecificationVo;
import com.matrix.md.service.ComponentService;
import com.matrix.md.service.InterService;
import com.matrix.md.service.SpecService;
import com.matrix.md.websocket.SocketData;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * SpecController.
 *
 * @author Matrix <xhyrzldf@gmail.com>
 * @since 2021/12/1 at 11:57 上午
 * Suffering is the most powerful teacher of life.
 */
@RestController
@RequestMapping("/specs")
@Api(tags = "规范接口")
public class SpecController {

    @Autowired
    private SpecService specService;

    @Autowired
    private ComponentService comService;

    @Autowired
    private InterService interService;

    /**
     * 根据specId查询规范
     */
    @GetMapping("/{specId}")
    @ApiOperation("根据specId查询规范")
    public ResponseEntity<Specification> getSpecById(@ApiParam("规范id") @PathVariable("specId") String specId) {
        Specification spec = specService.getById(specId);
        Component component = comService.getById(spec.getCurrentComId());
        return ResponseEntity.ok(spec.toVo(component == null ? "无绑定组件" : component.getName()));
    }

    /**
     * 根据规范id获得规范
     *
     * @param page 页码
     * @return 规范列表
     */
    @GetMapping
    @ApiOperation(value = "分页获取所有规范")
    public ResponseEntity<IPage<SpecificationVo>> findAll(@RequestParam int page) {
        IPage<SpecificationVo> result = specService.page(Page.of(page, 10), Wrappers.emptyWrapper())
                .convert(s -> {
                    Component component = comService.getById(s.getCurrentComId());
                    return s.toVo(component == null ? "无绑定组件" : component.getName());
                });
        return ResponseEntity.ok(result);
    }

    /**
     * 根据规范id获得规范关联的组件
     *
     * @param specId 规范id
     * @return 组件列表
     */
    @GetMapping("/components")
    @ApiOperation(value = "根据规范id获得规范关联的组件")
    public ResponseEntity<List<Component>> findComponents(@RequestParam int specId) {
        String idString = String.valueOf(specId);
        List<Component> componentList = comService.list(Wrappers.lambdaQuery(Component.class)
                .eq(Component::getSpecAbleIds, idString));
        return ResponseEntity.ok(componentList);
    }

    /**
     * 更改规范绑定的组件id
     *
     * @param specId      规范id
     * @param componentId 要新绑定的组件id
     * @return 是否成功
     */
    @PutMapping
    @ApiOperation(value = "更改规范绑定的组件id")
    public ResponseEntity<String> updateComponents(
            @RequestParam @ApiParam(name = "specId", value = "规范id") long specId,
            @RequestParam @ApiParam(name = "componentId", value = "要新绑定的组件id") long componentId) {
        boolean result = specService.lambdaUpdate()
                .eq(Specification::getId, specId)
                .set(Specification::getCurrentComId, componentId)
                .update();
        SocketData.updateActiveComponent(componentId);
        return ResponseEntity.ok(result ? "更新成功" : "更新失败");
    }

    /**
     * 解除规范绑定的组件
     *
     * @param specId 规范id
     * @return 是否成功
     */
    @DeleteMapping("/components")
    @ApiOperation(value = "解除规范绑定的组件")
    public ResponseEntity<String> deleteComponents(@RequestParam @ApiParam(name = "specId", value = "规范id") long specId) {
        boolean result = specService.lambdaUpdate()
                .eq(Specification::getId, specId)
                .set(Specification::getCurrentComId, 0)
                .update();
        SocketData.updateActiveComponent(0);
        return ResponseEntity.ok(result ? "更新成功" : "更新失败");
    }

    /**
     * 根据规范id获得关联的接口Vo
     *
     * @param specId 规范id
     * @return 接口Vo列表
     */
    @GetMapping("/interfaces")
    @ApiOperation(value = "根据规范id获得关联的接口Vo")
    public ResponseEntity<List<InterVo>> getInterVos(@RequestParam long specId) {
        List<InterVo> interVos = interService.getInterfaceBySpecId(specId);
        return ResponseEntity.ok(interVos);
    }
}
