package com.matrix.md.websocket;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.websocket.*;
import javax.websocket.server.ServerEndpoint;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * WebSocket component
 *
 * @author Matrix <xhyrzldf@gmail.com>
 * @since 2021/12/2 at 8:07 PM
 * Suffering is the most powerful teacher of life.
 */
@ServerEndpoint(value = "/message")
@Component
@Slf4j
public class WebSocket {

    /**
     * 用来存储每个会话的session,静态的不会被实例化
     */
    public static CopyOnWriteArraySet<WebSocket> webSocketSets = new CopyOnWriteArraySet<>();

    /**
     * 每个浏览器连接都会有一个新的会话对象
     */
    private Session session;

    /**
     * 所有客户端发送消息
     *
     * @param message 消息
     */
    public static void sendMessage(String message) {
        for (WebSocket ws : webSocketSets) {
            ws.session.getAsyncRemote().sendText(message);
        }
    }

    /**
     * 主要用来监听连接建立，config用来获取WebsocketConfig中的配置信息
     *
     * @param session 会话
     * @param config  配置信息
     */
    @OnOpen
    public void onOpen(Session session, EndpointConfig config) {
        this.session = session;
        webSocketSets.add(this);
        log.info("[websocket消息] 有新的连接, 总数:{}", webSocketSets.size());
    }

    @OnClose
    public void onClose() {
        webSocketSets.remove(this);
        log.info("[websocket消息]连接断开, 总数:{}", webSocketSets.size());
    }

    @OnError
    public void onError(Throwable e, Session session) {
        webSocketSets.remove(this);
        log.info("[websocket消息]连接出错或未关闭socket：" + e.getMessage());

    }

    @OnMessage
    public void onMessage(String message, Session session) {
        for (WebSocket ws : webSocketSets) {
            ws.session.getAsyncRemote().sendText("广播：" + message);
        }
        log.info("[websocket消息]收到客户端发来的消息:{}", message);
    }

}
