package com.example.pt;

import com.example.pt.collectors.DataRecord;
import com.example.pt.collectors.DataRecordService;
import javafx.application.Application;
import javafx.collections.ListChangeListener;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.chart.CategoryAxis;
import javafx.scene.chart.LineChart;
import javafx.scene.chart.NumberAxis;
import javafx.scene.chart.XYChart;
import javafx.scene.control.*;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.StackPane;
import javafx.stage.Stage;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.context.ConfigurableApplicationContext;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * ChartApp.
 *
 * @author Matrix <xhyrzldf@gmail.com>
 * @since 2023/3/21 at 18:07
 * Suffering is the most powerful teacher of life.
 */

@SpringBootApplication
public class ChartApp extends Application {

    private static final int RID_FENG_GE = 9999;
    private static final int RID_DA_SI_MA = 606118;

    private ConfigurableApplicationContext springContext;
    @Resource
    private DataRecordService dataRecordService;
    private LineChart<String, Number> lineChart;

    public static void main(String[] args) {
        Application.launch(args);
    }

    @Override
    public void init() {
        springContext = SpringApplication.run(ChartApp.class);
        dataRecordService = springContext.getBean(DataRecordService.class);
    }

    @Override
    public void start(Stage primaryStage) {
        // 获取初始数据
        List<DataRecord> dataRecords = dataRecordService.list();

        // 过滤 rid 为 9999 和 606118 的记录
        Map<Integer, List<DataRecord>> ridToDataRecordsMap = dataRecords.stream().filter(record -> record.getRid() == RID_FENG_GE || record.getRid() == RID_DA_SI_MA).collect(Collectors.groupingBy(DataRecord::getRid));

        // 创建折线图
        CategoryAxis xAxis = new CategoryAxis();
        xAxis.setStyle("-fx-font-size: 1px;");
        NumberAxis yAxis = new NumberAxis();
        xAxis.setLabel("时间 (HH:mm)");
        yAxis.setLabel("票数增长量");

        lineChart = new LineChart<>(xAxis, yAxis);
        lineChart.setTitle("票数增长量统计");

        // 添加时间区间下拉框和按钮
        ComboBox<Integer> intervalComboBox = new ComboBox<>();
        intervalComboBox.getItems().addAll(1, 5, 10, 30, 60);
        intervalComboBox.getSelectionModel().selectFirst();

        Button updateButton = new Button("更新");
        updateButton.setOnAction(event -> updateChartData(ridToDataRecordsMap, intervalComboBox.getValue()));

        HBox hbox = new HBox(10, intervalComboBox, updateButton);
        hbox.setAlignment(Pos.CENTER);

        // 设置场景并显示图表
        BorderPane root = new BorderPane(lineChart);
        root.setTop(hbox);
        Scene scene = new Scene(root, 800, 600);
        primaryStage.setScene(scene);
        primaryStage.setTitle("票数增长量统计");
        primaryStage.show();

        // 初始化图表数据
        updateChartData(ridToDataRecordsMap, intervalComboBox.getValue());
    }

    private void updateChartData(Map<Integer, List<DataRecord>> ridToDataRecordsMap, int intervalInMinutes) {
        lineChart.getData().clear();

        for (Map.Entry<Integer, List<DataRecord>> entry : ridToDataRecordsMap.entrySet()) {
            Integer rid = entry.getKey();
            List<DataRecord> records = entry.getValue();

            // 按时间排序
            List<DataRecord> sortedRecords = new ArrayList<>(records);
            sortedRecords.sort(Comparator.comparing(DataRecord::getTimestamp));

            // 计算票数增长量
            List<DataRecord> aggregatedData = new ArrayList<>();
            DataRecord previousRecord = null;
            LocalDateTime currentIntervalStart = sortedRecords.get(0).getTimestamp();
            int maxScInCurrentInterval = sortedRecords.get(0).getSc();

            for (DataRecord record : sortedRecords) {
                if (record.getTimestamp().isAfter(currentIntervalStart.plus(intervalInMinutes, ChronoUnit.MINUTES))) {
                    // 当前记录超出当前时间区间，将之前的数据添加到折线图
                    int scGrowth = previousRecord == null ? maxScInCurrentInterval : maxScInCurrentInterval - previousRecord.getSc();
                    aggregatedData.add(new DataRecord(null, rid, scGrowth, currentIntervalStart));

                    // 更新时间区间和最大票数
                    previousRecord = new DataRecord(null, rid, maxScInCurrentInterval, currentIntervalStart);
                    currentIntervalStart = currentIntervalStart.plus(intervalInMinutes, ChronoUnit.MINUTES);
                    maxScInCurrentInterval = record.getSc();
                } else {
                    // 更新当前时间区间的最大票数
                    maxScInCurrentInterval = Math.max(maxScInCurrentInterval, record.getSc());
                }
            }

            // 添加最后一个时间区间的数据
            int lastScGrowth = previousRecord == null ? maxScInCurrentInterval : maxScInCurrentInterval - previousRecord.getSc();
            aggregatedData.add(new DataRecord(null, rid, lastScGrowth, currentIntervalStart));

            // 将数据添加到折线图
            XYChart.Series<String, Number> series = new XYChart.Series<>();
            series.setName(rid == RID_FENG_GE ? "峰哥" : "大司马");

            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
            boolean isFirstDataPoint = true;
            for (DataRecord record : aggregatedData) {
                if (!isFirstDataPoint) {
                    String formattedTimestamp = record.getTimestamp().format(formatter);
                    XYChart.Data<String, Number> dataPoint = new XYChart.Data<>(formattedTimestamp, record.getSc());
                    series.getData().add(dataPoint);
                    dataPoint.nodeProperty().addListener((observable, oldValue, newValue) -> {
                        if (newValue != null) {
                            newValue.addEventHandler(MouseEvent.MOUSE_CLICKED, event -> {
                                String tooltipText = "时间: " + formattedTimestamp + "\n票数增长量: " + dataPoint.getYValue();
                                Alert alert = new Alert(Alert.AlertType.INFORMATION, tooltipText, ButtonType.OK);
                                alert.setHeaderText(null);
                                alert.setTitle("票数增长详情");
                                alert.show();
                            });
                        }
                    });
//                    dataPoint.nodeProperty().addListener((observable, oldValue, newValue) -> {
//                        if (newValue != null) {
//                            StackPane stackPane = (StackPane) newValue;
//                            Label label = new Label(String.valueOf(dataPoint.getYValue()));
//                            stackPane.getChildren().add(label);
//                        }
//                    });
                } else {
                    isFirstDataPoint = false;
                }
            }


            lineChart.getData().add(series);
        }
    }

    @Override
    public void stop() {
        springContext.close();
    }
}


