package com.matrix.md.sdb.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.matrix.md.common.DBSource;
import com.matrix.md.core.dao.OperateTableMapper;
import com.matrix.md.sdb.entity.ColumnInfo;
import com.matrix.md.sdb.entity.DatabaseInfo;
import com.matrix.md.sdb.entity.PlatformInfo;
import com.matrix.md.sdb.entity.TableInfo;
import com.matrix.md.sdb.entity.vo.ColumnVo;
import com.matrix.md.sdb.entity.vo.PlatformVo;
import com.matrix.md.sdb.entity.vo.TableVo;
import com.matrix.md.sdb.mapper.*;
import com.matrix.md.sdb.service.IOperateTableService;
import com.matrix.md.xdb.mapper.OperateXdbTableMapper;
import com.matrix.md.ydb.mapper.OperateYdbTableMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiFunction;

import static java.util.stream.Collectors.toList;

/**
 * IOperateTableServiceImpl.
 * 相同的代码，操作的的数据源是SDB数据源
 *
 * @author Matrix <xhyrzldf@gmail.com>
 * @since 2020/5/20 at 3:30 下午
 */
@SuppressWarnings("DuplicatedCode")
@Slf4j
@Service
public class IOptTableServiceImpl implements IOperateTableService {

    @Autowired
    private OperateSdbTableMapper operateSdbTableMapper;

    @Autowired
    private OperateXdbTableMapper operateXdbTableMapper;

    @Autowired
    private OperateYdbTableMapper operateYdbTableMapper;

    @Autowired
    private PlatformInfoMapper platformInfoMapper;

    @Autowired
    private DatabaseInfoMapper databaseInfoMapper;

    @Autowired
    private TableInfoMapper tableInfoMapper;

    @Autowired
    private ColumnInfoMapper columnInfoMapper;


    @Override
    public List<TableInfo> dynamicCreateTables(PlatformVo platformVo, DBSource dbSource) {
        //1.拆解vo对象，将其转换成各个mapper需要的对象
        //根据name查询platform id 以及其DB信息
        Long platformId = platformInfoMapper.selectOne(Wrappers.<PlatformInfo>lambdaQuery()
                .eq(PlatformInfo::getName, platformVo.getName()))
                .getId();

        Long databaseId = databaseInfoMapper.selectOne(Wrappers.<DatabaseInfo>lambdaQuery()
                .eq(DatabaseInfo::getPlatformId, platformId))
                .getId();

        //操作根据结构进行操作
        List<Long> tableIds = new ArrayList<>();
        for (TableVo tableParams : platformVo.getTableList()) {
            //创建表
            operateBySource(dbSource, tableParams, OperateTableMapper::createNewTable);
            log.info("成功创建了新表,表信息为{}", tableParams);

            //记录表结构信息
            TableInfo tableInfo = tableParams.trans2Table(1)
                    .setDatabaseId(databaseId);
            tableInfoMapper.insert(tableInfo);
            tableIds.add(tableInfo.getId());

            List<ColumnInfo> columnInfoList = tableParams.getColumnVoList().stream()
                    .map(ColumnVo::tran2Column)
                    .map(columnInfo -> columnInfo.setTableInfoId(tableInfo.getId()))
                    .collect(toList());

            //记录列结构信息
            columnInfoList.forEach(columnInfoMapper::insert);
        }

        //查询tableIds的表数据并返回 (手动将tableInfoId设为NULL避免信息冗余,完整应当建立一个DTO/VO对象来操作)
        return tableInfoMapper.selectBatchIds(tableIds);
    }

    @Override
    public int dynamicDeleteTables(String tableName, DBSource dbSource) {
        //根据表名找到表id
        Long tableId = tableInfoMapper.selectOne(Wrappers.<TableInfo>lambdaQuery()
                .eq(TableInfo::getName, tableName))
                .getId();

        //删除列信息与表信息
        int delColumnLines = columnInfoMapper.delete(Wrappers.<ColumnInfo>lambdaQuery()
                .eq(ColumnInfo::getTableInfoId, tableId));
        log.info("成功删除了{}表的结构信息,删除列的数量为{}", tableName, delColumnLines);
        int delTableLines = tableInfoMapper.deleteById(tableId);
        log.info("成功删除了{}表的结构信息,删除表的数量为{}", tableName, delTableLines);

        operateBySource(dbSource, tableName, OperateTableMapper::dropTable);
        log.info("成功删除了{}表", tableName);
        return delColumnLines;
    }


    /**
     * 根据不同的数据源进行不同的数据库操作
     *
     * @param dbSource     要操作的数据源
     * @param param        参数
     * @param optFunctions 要进行的数据库操作
     * @param <R>          返回值类型
     * @param <V>          参数类型
     * @return function操作返回的T类型对象
     */
    private <V, R> R operateBySource(DBSource dbSource, V param, BiFunction<OperateTableMapper, V, R> optFunctions) {
        switch (dbSource) {
            case XDB:
                return optFunctions.apply(operateXdbTableMapper, param);
            case YDB:
                return optFunctions.apply(operateYdbTableMapper, param);
            default:
            case SDB:
                return optFunctions.apply(operateSdbTableMapper, param);
        }
    }
}
