package com.tykj.zydd.misc.exception;



import com.tykj.zydd.misc.base.ResponseObj;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;

/**
 * 错误处理类
 * 所有的报错信息都会通过本层的方法向外界返回
 *
 * @author HuangXiahao
 **/
@RestControllerAdvice
@Slf4j
public class GlobalExceptionHandler {
    /**
     * 报错信息都会通过这个方法处理并通过统一的返回方式进行返回
     *
     * @param e 报错信息
     */
    @ExceptionHandler(Exception.class)
    public ResponseObj errorMessage(Exception e) {
        log.error("[其他异常]", e);
        e.printStackTrace();
        return new ResponseObj(e.getMessage(),HttpStatus.INTERNAL_SERVER_ERROR+"");
    }

    /**
     * 参数校验错误
     * @param e 报错信息
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseObj errorMessage(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String errorMessage = "参数校验失败:";
        for (FieldError fieldError : bindingResult.getFieldErrors()) {
            errorMessage += fieldError.getDefaultMessage() + ", ";
        }
        log.error("[参数校验异常] {}", errorMessage);
//        e.printStackTrace();
        return new ResponseObj(errorMessage,HttpStatus.BAD_REQUEST+"");
    }

    /**
     * 业务错误
     *
     * @param e 报错信息
     */
    @ExceptionHandler(ApiException.class)
    public ResponseObj errorMessage(ApiException e) {
        log.warn("[自定义异常]"+e.getResponseObj().getMessage());
        return e.getResponseObj();
    }

    /**
     * 数据合法性验证报错会通过这个方法处理并通过统一的返回方式进行返回
     *
     * @param e 报错信息
     */
    @ExceptionHandler(BindException.class)
    public ResponseObj errorMessage(BindException e) {
        log.error("[参数异常] 检测到用户访问接口没有提供正确的参数 {}", e.toString());
        e.printStackTrace();
        BindingResult bindingResult = e.getBindingResult();
        String message = null;
        if (bindingResult.hasErrors()) {
            FieldError fieldError = bindingResult.getFieldError();
            if (fieldError != null) {
                message = fieldError.getField() + fieldError.getDefaultMessage();
            }
        }
        return new ResponseObj(message,HttpStatus.BAD_REQUEST+"");
    }


}

