package com.tykj.zydd.misc.util;

import cn.hutool.core.util.ObjectUtil;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.beans.BeansException;
import org.springframework.util.CollectionUtils;

import java.beans.PropertyDescriptor;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @author : sc
 */
public class BeanCopyUtil {
    /**
     * 拷贝非空属性
     *
     * @param source 来源
     * @param target 目标
     * @throws BeansException 拷贝异常
     */
    public static void copyProperties(Object source, Object target) throws BeansException {
        copyProperties(source, target, (String[]) null);
    }


    /**
     * 拷贝非空属性
     *
     * @param source           来源
     * @param target           目标
     * @param ignoreProperties 拷贝时忽略的属性
     * @throws BeansException 拷贝异常
     */
    public static void copyProperties(Object source, Object target, String... ignoreProperties) throws BeansException {
        org.springframework.beans.BeanUtils.copyProperties(source, target, ignoreProperties);
    }

    /**
     * 拷贝创建新对象
     *
     * @param source 来源
     * @param supplier 目标
     * @throws BeansException 拷贝异常
     */
    public static <T> T copyProperties(Object source, Supplier<T> supplier) {
        T t =  supplier.get();
        copyProperties(source, t);
        return t;
    }


    /**
     * 拷贝非空属性
     *
     * @param source 来源
     * @param target 目标
     * @throws BeansException 拷贝异常
     */
    public static void copyNotNullProperties(Object source, Object target) throws BeansException {
        copyProperties(source, target, getNullPropertyNames(source));
    }

    /**
     * 列表拷贝
     *
     * @param sources  来源集合
     * @param supplier 获取目标对象实例
     * @param <T>      目标对象
     * @return 列表拷贝至目标对象列表
     */
    public static <T> List<T> copyCollectionObjects(Collection<?> sources, Supplier<T> supplier) {
        if (CollectionUtils.isEmpty(sources)) {
            return Collections.emptyList();
        }
        return sources.stream().map(source -> {
            T t = supplier.get();
            copyProperties(source, t);
            return t;
        }).collect(Collectors.toList());
    }


    public static <T> List<T> copyCollectionObjects(Collection<?> sources, final Class<T> clazz) {
        if (CollectionUtils.isEmpty(sources)) {
            return Collections.emptyList();
        }
        return sources.stream().map(source -> {
            T t = null;
            try {
                t = clazz.newInstance();
                copyProperties(source, t);
            } catch (ReflectiveOperationException roe) {
            }
            return t;
        }).collect(Collectors.toList());
    }


    /**
     * 获取需要忽略的属性
     *
     * @param source
     * @return
     */
    private static String[] getNullPropertyNames(Object source) {
        final BeanWrapper src = new BeanWrapperImpl(source);
        PropertyDescriptor[] pds = src.getPropertyDescriptors();

        Set<String> emptyNames = new HashSet<>();
        for (PropertyDescriptor pd : pds) {
            Object srcValue = src.getPropertyValue(pd.getName());
            // 此处判断可根据需求修改
            if (ObjectUtil.isEmpty(srcValue)) {
                emptyNames.add(pd.getName());
            }
        }
        String[] result = new String[emptyNames.size()];
        return emptyNames.toArray(result);
    }


}
