package com.tykj.zydd.misc.util;

import cn.hutool.core.util.ObjectUtil;
import com.tykj.zydd.misc.exception.ApiException;


import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;


public class DateUtil {

    public static LocalDateTime getLocalDateTime(Date date){
        Instant instant = date.toInstant();
        return instant.atZone(ZoneOffset.of("+8")).toLocalDateTime();
    }

    public static LocalDate getLocalDate(Date date){
        Instant instant = date.toInstant();
        ZoneId zoneId = ZoneId.of("GMT+8");
        LocalDateTime localDateTime = instant.atZone(zoneId).toLocalDateTime();
        return localDateTime.toLocalDate();
    }

    public static LocalDate string2LocalDate(String format){
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        return  LocalDate.parse(format, formatter);
    }

    public static LocalDate string2LocalDateTime(String format){
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        return  LocalDate.parse(format, formatter);
    }

    public static String localDate2String(LocalDate localDate){
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        return formatter.format(localDate);
    }

    public static String localDateTime2String2(LocalDateTime localDateTime){
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        return formatter.format(localDateTime);
    }

    public static String localDateTime2String(LocalDateTime localDateTime){
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        return formatter.format(localDateTime);
    }

    /**
     * 时间转换成年月日时分秒
     * @param localDateTime
     * @return
     */
    public static String localDateTime2No(LocalDateTime localDateTime){
        String year = localDateTime.getYear()+"";
        String monthValue = localDateTime.getMonthValue()+"";
        if (monthValue.length() == 1){
            monthValue = "0"+monthValue;
        }
        String dayOfMonth = localDateTime.getDayOfMonth()+"";
        if (dayOfMonth.length() == 1){
            dayOfMonth = "0"+dayOfMonth;
        }
        //时分秒
        String hour = localDateTime.getHour()+"";
        if (hour.length() == 1){
            hour = "0"+hour;
        }
        String minute = localDateTime.getMinute()+"";
        if (minute.length() == 1){
            minute = "0"+minute;
        }
        String second = localDateTime.getSecond()+"";
        if (second.length() == 1){
            second = "0"+second;
        }
        return year+monthValue+dayOfMonth+hour+minute+second;
    }

    /**
     * 将时间戳转换成LocalDateTime  秒
     * @param timestamp
     * @return
     */
    public static LocalDateTime convertTimestamp(long timestamp) {
        Instant instant = Instant.ofEpochSecond(timestamp);
        return LocalDateTime.ofInstant(instant, ZoneOffset.of("+8"));
    }


    /**
     * 判断当天是周几
     */
    public  static  int  dayForWeek(Date date) {
        if(ObjectUtil.isEmpty(date)){
            return -1;
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd" );
        String time = format.format(date);
        Calendar c = Calendar.getInstance();
        try {
            c.setTime(format.parse(time));
        } catch (ParseException e) {
            return -1;
        }
        int  dayForWeek;
        if (c.get(Calendar.DAY_OF_WEEK) == 1 ){
            dayForWeek = 7 ;
        }else {
            dayForWeek = c.get(Calendar.DAY_OF_WEEK) - 1 ;
        }
        return  dayForWeek;
    }
    /**
     * 获取当天0点时间
     */
    public  static  Date  getDate() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取指定时间的23：59：59
     * @param time
     * @return
     */
    public static Date getEndOfDay(Date time) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return calendar.getTime();
    }

    /**
     * 获取指定时间的 00:00:00
     */
    public  static  Date  getStartDate(Date time) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }
    /**
     * 获取当前时间前的时间
     * type 类型 时分月等等
     * num
     */
    public  static  Date  getHistoryDate(Integer type,Integer num) {
        Calendar beforeTime = Calendar.getInstance();
        // n分钟之前的时间
        beforeTime.add(type, num);
        return beforeTime.getTime();
    }


    /**
     * 获取本周的第一天
     * @return String
     **/
    public static Date getWeekStart() {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(Calendar.MONDAY);
        cal.add(Calendar.WEEK_OF_MONTH, 0);
        cal.set(Calendar.DAY_OF_WEEK,Calendar.MONDAY);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }


    /**
     * 获取任意天后的时间
     *  day 1表示后一天 -1表示前一天
     */
    public static Date getDay(Date time, int day){
        try {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(time);
            calendar.add(Calendar.DAY_OF_MONTH, day);
            return calendar.getTime();
        }catch (Exception e){
            throw new ApiException("获取几天后日期错误");
        }
    }

    /**
     * 根据年月获取月初第一天日期
     * @param year  年
     * @param month 月
     * @return
     */
    public static String getFirstDay(int year,int month,String format) {
        Calendar cale = Calendar.getInstance();
        cale.set(Calendar.YEAR,year);
        cale.set(Calendar.MONTH, month-1);
        //获取月最小天数
        int minDay = cale.getActualMinimum(Calendar.DAY_OF_MONTH);
        cale.set(Calendar.DAY_OF_MONTH, minDay);
        cale.set(Calendar.HOUR_OF_DAY, 0);
        cale.set(Calendar.MINUTE, 0);
        cale.set(Calendar.SECOND,0);
        //cale.set(Calendar.MILLISECOND, 0);
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(cale.getTime());
    }

    /**
     * 根据年月获取月末最后一天日期
     * @param year  年
     * @param month 月
     * @return
     */
    public static String getLastDay(int year,int month,String format) {
        Calendar cale = Calendar.getInstance();
        cale.set(Calendar.YEAR,year);
        cale.set(Calendar.MONTH, month-1);
        //获取月最大天数
        int lastDay = cale.getActualMaximum(Calendar.DAY_OF_MONTH);
        cale.set(Calendar.DAY_OF_MONTH, lastDay);
        cale.set(Calendar.HOUR_OF_DAY, 23);
        cale.set(Calendar.MINUTE, 59);
        cale.set(Calendar.SECOND,59);
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(cale.getTime());
    }
    /**
     * 根据年月获取所有日期
     * @param year  年
     * @param month 月
     * @return
     */
    public static List<LocalDateTime> getAllDatesOfYearMonth(int year, Month month) {
        List<LocalDateTime> dates = new ArrayList<>();

        LocalDate firstDayOfMonth = LocalDate.of(year, month, 1);
        int daysInMonth = firstDayOfMonth.lengthOfMonth();

        for (int day = 1; day <= daysInMonth; day++) {
            LocalDateTime dateTime = LocalDateTime.of(year, month, day, 0, 0, 0);
            dates.add(dateTime);
        }

        return dates;
    }

    /**
     * 根据两个时间得到日期列表 根据类型判断是否要补到周一或周末
     * @param startDate  开始时间
     * @param endDate    结束时间
     * @param type   类型 是否要补全
     * @return  日期列表
     */
    public static List<LocalDateTime> completeDateList(LocalDateTime startDate, LocalDateTime endDate,Integer type) {
        List<LocalDateTime> completedDateList = new ArrayList<>();
        // 补全起始日期之前的数据
        if(ObjectUtil.isNotEmpty(type)) {
            while (startDate.getDayOfWeek() != DayOfWeek.MONDAY) {
                startDate = startDate.minusDays(1);
            }
            while (endDate.getDayOfWeek() != DayOfWeek.SUNDAY) {
                endDate = endDate.minusDays(-1);
            }
        }
        LocalDateTime tempDate = startDate;
        // 补全起始日期到结束日期的数据
        while (!tempDate.isAfter(endDate)) {
            completedDateList.add(tempDate);
            tempDate = tempDate.plusDays(1);
        }
        return completedDateList;
    }

    /**
     * 根据时间获取年月日
     * @param date  时间
     * @return  年月日Map
     */
    public static Map<String,Integer> getDateDetail(Date date) {
        Map<String,Integer> map = new HashMap<>();
        if(ObjectUtil.isEmpty(date)){
            return map;
        }
        // 创建一个Calendar对象
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        // 获取年份
        int year = calendar.get(Calendar.YEAR);
        // 获取月份（注意，月份是从0开始的，所以我们需要加1）
        int month = calendar.get(Calendar.MONTH) + 1;
        // 获取日期
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        map.put("year",year);
        map.put("month",month);
        map.put("day",day);
        return map;
    }

    /**
     * 根据时间段 得到每月的第一天
     * @param startDateTime  开始时间
     * @param endDateTime    结束时间
     * @return 时间集合
     */
    public static List<LocalDateTime> getFirstDaysOfMonth(LocalDateTime startDateTime, LocalDateTime endDateTime) {
        List<LocalDateTime> result = new ArrayList<>();

        LocalDateTime currentDateTime = startDateTime;

        while (!currentDateTime.isAfter(endDateTime)) {
            LocalDateTime firstDayOfMonth = LocalDateTime.of(currentDateTime.getYear(), currentDateTime.getMonth(), 1, 0, 0);
            result.add(firstDayOfMonth);

            currentDateTime = currentDateTime.plusMonths(1);
        }

        return result;
    }
}
