package com.tykj.zydd.user.entity;

import org.springframework.beans.BeanUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

public class CompareUtil {

    /**
     * 比较两个对象相同属性的值是否相等，忽略指定属性
     *
     * @param obj1           对象1
     * @param obj2           对象2
     * @param ignoreProps    需要忽略的属性
     * @return  true 如果两个对象相同属性的值相等（忽略指定属性），否则返回 false
     */
    public static boolean compare(Object obj1, Object obj2, String... ignoreProps) throws Exception {
        if (obj1 == null || obj2 == null) {
            return obj1 == obj2;
        }
        Class<?> clazz = obj1.getClass();
        List<String> ignoreList = Arrays.asList(ignoreProps);
        List<Field> fieldList = new ArrayList<>();
        while (clazz != null && !clazz.getName().toLowerCase().equals("java.lang.object")) {
            fieldList.addAll(Arrays.asList(clazz.getDeclaredFields()));
            clazz = clazz.getSuperclass();
        }
        List<String> fieldNames1 = getFieldNames(obj1);
        List<String> fieldNames2 = getFieldNames(obj2);
        fieldNames1.retainAll(fieldNames2);
        fieldNames1.removeAll(ignoreList);
        for (String fieldName : fieldNames1) {
            PropertyDescriptor pd = new PropertyDescriptor(fieldName, obj1.getClass());
            Object value1 = pd.getReadMethod().invoke(obj1);
            Object value2 = pd.getReadMethod().invoke(obj2);
            if (value1 == null && value2 == null) {
                continue;
            }
            if (value1 == null || value2 == null || !value1.equals(value2)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 获取对象的字段名
     *
     * @param obj   对象
     * @return  List<String> 字段名列表
     */
    private static List<String> getFieldNames(Object obj) {
        Class<?> clazz = obj.getClass();
        List<Field> fieldList = new ArrayList<>();
        while (clazz != null && !clazz.getName().toLowerCase().equals("java.lang.object")) {
            fieldList.addAll(Arrays.asList(clazz.getDeclaredFields()));
            clazz = clazz.getSuperclass();
        }
        return fieldList.stream().map(Field::getName).collect(Collectors.toList());
    }

    /**
     * 将 Entity 对象转化为 DTO 对象，并比较两个 DTO 对象相同属性的值是否相等
     *
     * @param entity    Entity 对象
     * @param dtoClass  DTO 类的 Class 对象
     * @param dto       DTO 对象
     * @return  true 如果 Entity 和 DTO 对象相同属性的值相等，否则返回 false
     */
    public static <T> boolean compareEntityToDto(T entity, Class<T> dtoClass, T dto) throws Exception {
        if (entity == null || dto == null) {
            return entity == dto;
        }
        T temp = dtoClass.newInstance();
        BeanUtils.copyProperties(entity, temp);
        return compare(temp, dto);
    }
}