package com.tykj.index.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.tykj.index.entity.IndexInfo;
import com.tykj.index.entity.vo.*;
import com.tykj.index.repository.IndexInfoRepository;
import com.tykj.model.entity.Rule;
import com.tykj.model.entity.vo.ColumnVO;
import com.tykj.model.entity.vo.DelTableVO;
import com.tykj.model.entity.vo.TableVO;
import com.tykj.model.service.ModelService;
import com.tykj.model.utils.MonthUtil;
import com.tykj.model.utils.SessionUtil;
import org.apache.logging.log4j.util.Strings;
import org.hibernate.internal.SessionImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.sql.SQLException;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

@SuppressWarnings("Duplicates")
@Service
public class IndexInfoService {

    @Autowired
    private IndexInfoRepository indexInfoRepository;
    @Autowired
    private ModelService modelService;
    @Autowired
    private JdbcTemplate jdbcTemplate;
    @Autowired
    private SessionUtil sessionUtil;

    private final List<String> areas = Lists.newArrayList("杭州", "宁波", "衢州");

    private DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM");

    public void save(IndexInfoVo indexInfoVo) {
        //参数检查
        if (nonNull(indexInfoVo.getId())) {
            throw new RuntimeException("新增操作不可附带id");
        }
        if (isNull(indexInfoVo.getAlias())) {
            throw new RuntimeException("别名不能为空");
        }
        if (nonNull(indexInfoVo.getParentId())) {
            Double weight = indexInfoVo.getWeight();
            //查出其他同parentId指标的权重合
            List<IndexInfo> indexInfos = indexInfoRepository.findAllByParentId(indexInfoVo.getParentId());
            double sum = indexInfos.stream()
                    .mapToDouble(IndexInfo::getWeight)
                    .sum();
            if (weight == null || weight < 0 || sum + weight > 1) {
                throw new RuntimeException("权重值非法");
            }
        }
        //新增时建立模型实体
        Integer tableId = modelService.newTable(createTableVo(indexInfoVo.getAlias())).getId();
        //保存信息
        IndexInfo indexInfo = indexInfo(indexInfoVo, tableId);
        indexInfoRepository.save(indexInfo);
        //将sql语句查出来的数据插入到指标对应的表
        handleData(indexInfo.getAlias(), indexInfo.getSqlContent());
        updateDataValue(indexInfo);
    }

    public void update(IndexInfoVo indexInfoVo) {
        //参数检查
        if (isNull(indexInfoVo.getId())) {
            throw new RuntimeException("修改操作必须附带id");
        }
        if (nonNull(indexInfoVo.getParentId())) {
            Double weight = indexInfoVo.getWeight();
            //查出其他同parentId指标的权重合
            List<IndexInfo> indexInfos = indexInfoRepository.findAllByParentId(indexInfoVo.getParentId());
            double sum = indexInfos.stream()
                    .filter(indexInfo -> !Objects.equals(indexInfo.getId(), indexInfoVo.getId()))
                    .mapToDouble(IndexInfo::getWeight)
                    .sum();
            if (weight == null || weight < 0 || sum + weight > 1) {
                throw new RuntimeException("权重值非法");
            }
        }
        //重新对应已建立的模型实体
        Integer tableId = indexInfoRepository.findById(indexInfoVo.getId())
                .map(IndexInfo::getTableId)
                .orElseThrow(() -> new RuntimeException("id为[" + indexInfoVo.getId() + "]的数据不存在"));
        //保存信息
        IndexInfo indexInfo = indexInfo(indexInfoVo, tableId);
        indexInfo.setId(indexInfoVo.getId());
        indexInfoRepository.save(indexInfo);
        //将sql语句查出来的数据插入到指标对应的表
        handleData(indexInfo.getAlias(), indexInfo.getSqlContent());
        updateDataValue(indexInfo);
    }

    /**
     * 根据id更新其上级节点数据的值
     */
    public void updateValue(Integer id) {
        IndexInfo indexInfo = indexInfoRepository.findById(id)
                .orElseThrow(() -> new RuntimeException("id为[" + id + "]的数据不存在"));
        updateDataValue(indexInfo);
    }

    /**
     * 查询整个列表
     *
     * @return 返回的是由根节点展开的指标数据
     */
    public List<IndexInfoVo> findAll() {
        //先查询根节点(默认parentId为null的为根节点) 再根据id和parentId递归查询
        return indexInfoRepository.findAllByParentId(null).stream()
                .map(this::indexInfoVo)
                .collect(Collectors.toList());
    }

    /**
     * 根据parentId查询剩余可用的权重值
     */
    public Double checkRemainWeight(Integer parentId) {
        double sum = indexInfoRepository.findAllByParentId(parentId).stream()
                .mapToDouble(IndexInfo::getWeight)
                .sum();
        BigDecimal result = BigDecimal.valueOf(1).add(BigDecimal.valueOf(sum).negate());
        return result.doubleValue();
    }

    /**
     * 查询图表数据
     */
    @SuppressWarnings("Duplicates")
    public ChartVO querySQL(Integer id) {
        IndexInfoVo indexInfoVo = indexInfoRepository.findById(id)
                .map(this::indexInfoVo)
                .orElseThrow(() -> new RuntimeException("id为[" + id + "]的数据不存在"));
        List<IndexInfo> children = indexInfoRepository.findAllByParentId(id);
        //申明返回值
        List<String> names = new ArrayList<>();
        List<Double> values = new ArrayList<>();
        List<Double> values2 = new ArrayList<>();
        List<String> rankNames = new ArrayList<>();
        List<String> rankColors = new ArrayList<>();
        List<AreaChart> areaCharts = new ArrayList<>();
        //总数据
        List<Map<String, Object>> data = new ArrayList<>();
        try {
            data = modelService.findAllByName(indexInfoVo.getAlias());
        } catch (SQLException e) {
            e.printStackTrace();
        }
        //根据时间划分
        List<String> times = MonthUtil.getMountListEndNow("2021-01");
        List<Rank> ranks = indexInfoVo.getRanks();
        if (children.isEmpty()) {
            //没有子指标
            names.addAll(times);
            for (String area : areas) {
                List<Double> areaValues = new ArrayList<>();
                List<Double> areaValues2 = new ArrayList<>();
                List<String> areaRankNames = new ArrayList<>();
                List<String> areaRankColors = new ArrayList<>();
                for (String time : times) {
                    //计算平均值
                    double average = data.stream()
                            .filter(entity -> Objects.equals(getTime(entity), time))
                            .filter(entity -> Objects.equals(getArea(entity), area))
                            .filter(entity -> nonNull(entity.get("value")))
                            .mapToDouble(entity -> (double) entity.get("value"))
                            .average()
                            .orElse(0);
                    //计算平均值2
                    double average2 = data.stream()
                            .filter(entity -> Objects.equals(getTime(entity), time))
                            .filter(entity -> Objects.equals(getArea(entity), area))
                            .filter(entity -> nonNull(entity.get("value2")))
                            .mapToDouble(entity -> (double) entity.get("value2"))
                            .average()
                            .orElse(0);
                    areaValues.add(average);
                    areaValues2.add(average2);
                    Rank rank = matchRank(average, ranks);
                    areaRankNames.add(rank.getName());
                    areaRankColors.add(rank.getColor());
                }
                //计算匹配范围
                AreaChart areaChart = new AreaChart(area, areaValues, areaValues2, areaRankNames, areaRankColors);
                areaCharts.add(areaChart);
            }
        } else {
            for (String time : times) {
                names.add(time);
                //计算平均值
                double average = data.stream()
                        .filter(entity -> Objects.equals(getTime(entity), time))
                        .filter(entity -> nonNull(entity.get("value")))
                        .mapToDouble(entity -> (double) entity.get("value"))
                        .average()
                        .orElse(0);
                //计算平均值2
                double average2 = data.stream()
                        .filter(entity -> Objects.equals(getTime(entity), time))
                        .filter(entity -> nonNull(entity.get("value2")))
                        .mapToDouble(entity -> (double) entity.get("value2"))
                        .average()
                        .orElse(0);
                values.add(average);
                values2.add(average2);
                //计算匹配范围
                Rank rank = matchRank(average, ranks);
                rankNames.add(rank.getName());
                rankColors.add(rank.getColor());
            }
        }
        return new ChartVO(names, values, values2, rankNames, rankColors, areaCharts);
    }


    /**
     * 查询开始时间和结束时间 查询期间的多个图表数据
     */
    public List<MouthChartVo> queryMouthCharts(Integer id, String start, String end) {
        return MonthUtil.getMonthList(start, end).stream()
                .map(month -> queryMouthChart(id, month))
                .collect(Collectors.toList());
    }

    /**
     * 查询指定月图表数据
     */
    @SuppressWarnings("Duplicates")
    private MouthChartVo queryMouthChart(Integer id, String mouth) {
        List<IndexInfo> children = indexInfoRepository.findAllByParentId(id);
        IndexInfoVo indexInfoVo = indexInfoRepository.findById(id)
                .map(this::indexInfoVo)
                .orElseThrow(() -> new RuntimeException("id为[" + id + "]的数据不存在"));
        //查询并过滤出本月的数据
        List<Map<String, Object>> data = new ArrayList<>();
        try {
            data = modelService.findAllByName(indexInfoVo.getAlias()).stream()
                    .filter(entity -> isInMouth(getTime(entity), mouth))
                    .collect(Collectors.toList());
        } catch (SQLException e) {
            e.printStackTrace();
        }
        //申明返回值
        List<String> names = new ArrayList<>();
        List<Double> values = new ArrayList<>();
        List<Double> values2 = new ArrayList<>();
        List<String> rankNames = new ArrayList<>();
        List<String> rankColors = new ArrayList<>();
        if (children.isEmpty()) {
            //没有子指标则根据区域划分
            for (String area : areas) {
                //计算平均值
                double average = data.stream()
                        .filter(entity -> Objects.equals(getArea(entity), area))
                        .filter(entity -> nonNull(entity.get("value")))
                        .mapToDouble(entity -> (double) entity.get("value"))
                        .filter(Objects::nonNull)
                        .average()
                        .orElse(0);
                //计算平均值2
                double average2 = data.stream()
                        .filter(entity -> Objects.equals(getArea(entity), area))
                        .filter(entity -> nonNull(entity.get("value2")))
                        .mapToDouble(entity -> (double) entity.get("value2"))
                        .filter(Objects::nonNull)
                        .average()
                        .orElse(0);
                names.add(area);
                values.add(average);
                values2.add(average2);
                //匹配符合的范围
                List<Rank> ranks = indexInfoVo.getRanks();
                Rank rank = matchRank(average, ranks);
                rankNames.add(rank.getName());
                rankColors.add(rank.getColor());
            }
        } else {
            //有子指标则根据子指标名称划分
            for (IndexInfo child : children) {
                //计算平均值
                List<Map<String, Object>> childData = new ArrayList<>();
                try {
                    childData = modelService.findAllByName(child.getAlias());
                } catch (SQLException e) {
                    e.printStackTrace();
                }
                double average = childData.stream()
                        .filter(entity -> nonNull(entity.get("value")))
                        .mapToDouble(entity -> (double) entity.get("value"))
                        .average()
                        .orElse(0);
                //计算平均值2
                double average2 = childData.stream()
                        .filter(entity -> nonNull(entity.get("value2")))
                        .mapToDouble(entity -> (double) entity.get("value2"))
                        .average()
                        .orElse(0);
                names.add(child.getLabel());
                values.add(average);
                values2.add(average2);
                //匹配符合的范围
                List<Rank> ranks = indexInfoVo.getRanks();
                Rank rank = matchRank(average, ranks);
                rankNames.add(rank.getName());
                rankColors.add(rank.getColor());
            }
        }
        return new MouthChartVo(mouth, names, values, values2, rankNames, rankColors);
    }


    /**
     * 根据id删除指标
     * 同时也会删除对应的模型实体
     */
    public void deleteById(Integer id) {
        Integer tableId = indexInfoRepository.findById(id)
                .map(IndexInfo::getTableId)
                .orElseThrow(() -> new RuntimeException("id为[" + id + "]的数据不存在"));
        indexInfoRepository.deleteById(id);
        modelService.delTable(new DelTableVO(tableId, null));
    }

    //-----------------------------------------以下为私有方法------------------------------------------------------------//

    /**
     * 判断一个Date是否是在本月
     */
    private Boolean isInMouth(String date, String mouth) {
        TemporalAccessor dest = dateTimeFormatter.parse(date);
        TemporalAccessor time = dateTimeFormatter.parse(mouth);
        return Objects.equals(dest.get(ChronoField.MONTH_OF_YEAR), time.get(ChronoField.MONTH_OF_YEAR))
                && Objects.equals(dest.get(ChronoField.YEAR), time.get(ChronoField.YEAR));
    }

    /**
     * 将sql语句查出来的数据插入到指标对应的表
     */
    private void handleData(String name, String sql) {
        if (Strings.isNotBlank(sql)) {
            List<Map<String, Object>> data = jdbcTemplate.queryForList(sql);
            for (Map<String, Object> entity : data) {
                Map<String, Object> map = new HashMap<>();
                map.put(name, entity);
                modelService.operationValueByEntityName(map, (SessionImpl) sessionUtil.getSession(), null, null);
            }
        }
    }

    /**
     * 根据权重同步计算并更新数据值
     */
    private void updateDataValue(IndexInfo indexInfo) {
        IndexInfo parent = indexInfoRepository.findById(indexInfo.getParentId())
                .orElseThrow(() -> new RuntimeException("id为[" + indexInfo.getParentId() + "]的上级数据不存在"));
        //其他子节点
        List<IndexInfo> children = indexInfoRepository.findAllByParentId(indexInfo.getParentId()).stream()
                .filter(indexInfo1 -> !Objects.equals(indexInfo1.getId(), indexInfo.getId()))
                .collect(Collectors.toList());
        List<Map<String, Object>> data = new ArrayList<>();
        try {
            data = modelService.findAllByName(indexInfo.getAlias());
        } catch (SQLException e) {
            e.printStackTrace();
        }
        if (nonNull(parent.getAlias())) {
            modelService.deleteAllBytableName(parent.getAlias());
            for (Map<String, Object> datum : data) {
                //根据上级数据去匹配下级数据来计算
                double newValue = 0D;
                for (IndexInfo child : children) {
                    try {
                        double childValue = modelService.findAllByName(child.getAlias()).stream()
                                .filter(entity -> Objects.equals(entity.get("area"), datum.get("area")) && Objects.equals(entity.get("time"), datum.get("time")))
                                .filter(entity -> nonNull(entity.get("value")))
                                .findAny()
                                .map(entity -> (double) entity.get("value"))
                                .map(value -> value * child.getWeight())
                                .orElse(0D);
                        newValue += childValue;
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }
                }
                double selfValue = (double) datum.get("value");
                newValue += (selfValue * indexInfo.getWeight());
                datum.put("value", newValue);
                datum.remove("id");
                //更新重新计算后的数据
                Map<String, Object> map = new HashMap<>();
                map.put(parent.getAlias(), datum);
                modelService.operationValueByEntityName(map, (SessionImpl) sessionUtil.getSession(), null, null);
            }
        }
        if (nonNull(parent.getParentId())) {
            updateDataValue(parent);
        }
    }

    private IndexInfo indexInfo(IndexInfoVo indexInfoVo, Integer tableId) {
        String ranks = "[]";
        try {
            ranks = new ObjectMapper().writeValueAsString(indexInfoVo.getRanks());
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        return new IndexInfo(
                indexInfoVo.getLabel(),
                indexInfoVo.getAlias(),
                indexInfoVo.getDescription(),
                indexInfoVo.getSql(),
                indexInfoVo.getLevel(),
                indexInfoVo.getWeight(),
                ranks,
                indexInfoVo.getParentId(),
                tableId
        );
    }

    private IndexInfoVo indexInfoVo(IndexInfo indexInfo) {
        List<Rank> ranks = new ArrayList<>();
        try {
            ranks = new ObjectMapper().readValue(indexInfo.getRanks(), new TypeReference<List<Rank>>() {
            });
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        List<IndexInfoVo> children = indexInfoRepository.findAllByParentId(indexInfo.getId()).stream()
                .map(this::indexInfoVo)
                .collect(Collectors.toList());
        return new IndexInfoVo(
                indexInfo.getId(),
                indexInfo.getLabel(),
                indexInfo.getAlias(),
                indexInfo.getDescription(),
                indexInfo.getSqlContent(),
                indexInfo.getLevel(),
                indexInfo.getWeight(),
                ranks,
                children,
                indexInfo.getParentId()
        );

    }

    private TableVO createTableVo(String name) {
        ArrayList<ColumnVO> columns = Lists.newArrayList(
                new ColumnVO(
                        null,
                        1,
                        "java.lang.String",
                        "area",
                        Strings.EMPTY,
                        255,
                        new Rule(),
                        Strings.EMPTY,
                        new ArrayList<>()
                ),
                new ColumnVO(
                        null,
                        1,
                        "java.lang.String",
                        "area",
                        Strings.EMPTY,
                        255,
                        new Rule(),
                        Strings.EMPTY,
                        new ArrayList<>()
                ), new ColumnVO(
                        null,
                        1,
                        "java.util.Date",
                        "time",
                        Strings.EMPTY,
                        255,
                        new Rule(),
                        Strings.EMPTY,
                        new ArrayList<>()
                ), new ColumnVO(
                        null,
                        1,
                        "java.lang.Double",
                        "value",
                        Strings.EMPTY,
                        20,
                        new Rule(),
                        Strings.EMPTY,
                        new ArrayList<>()
                ), new ColumnVO(
                        null,
                        1,
                        "java.lang.Double",
                        "value2",
                        Strings.EMPTY,
                        20,
                        new Rule(),
                        Strings.EMPTY,
                        new ArrayList<>()
                )
        );
        return new TableVO(
                Strings.EMPTY,
                name,
                Strings.EMPTY,
                null,
                null,
                columns
        );
    }

    private String getArea(Map<String, Object> entity) {
        return String.valueOf(entity.get("area"));
    }

    private String getTime(Map<String, Object> entity) {
        if (nonNull(entity.get("time"))) {
            return dateTimeFormatter.format((TemporalAccessor) entity.get("time"));
        } else {
            return Strings.EMPTY;
        }
    }

    private Integer getTimeNumber(String time) {
        if (Strings.isNotEmpty(time)) {
            return Integer.valueOf(time.replace("-", ""));
        } else {
            return -1;
        }
    }

    private Boolean isInRank(Double value, Rank rank) {
        Double max = nonNull(rank.getMax()) ? rank.getMax() : 0;
        Double min = nonNull(rank.getMin()) ? rank.getMin() : 0;
        return value >= min && value < max;
    }

    private Rank matchRank(Double value, List<Rank> ranks) {
        return ranks.stream()
                .filter(rank -> isInRank(value, rank))
                .findAny()
                .orElse(new Rank(Strings.EMPTY, null, null, "#FFFFFF"));
    }
}
