package com.tykj.datawarehouse.model.utils;


import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;

import java.util.Map;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

/**
 * @author C
 */
@Slf4j
public class SqlUtil {

    @Autowired
    private Environment env;

    /**
     * 修改表名的SQL语句
     *
     * @param table    原表名
     * @param newTable 新表名
     * @return SQL语句
     */
    public static String renameTable(String table, String newTable) {
        return String.format("rename table %s to %s;", table, newTable);
    }

    /**
     * 增加单个字段的SQL语句
     *
     * @param table  表名
     * @param name   字段名
     * @param type   字段类型
     * @param length 字段长度
     * @return SQL语句
     */
    public static String addColumn(String table, String name, String type, Integer length, Map<String, String> sqlTypeMap) {
        type = sqlTypeMap.get(type);
        boolean hasValue = !Strings.isNullOrEmpty(table)
                && !Strings.isNullOrEmpty(name)
                && nonNull(type);
        StringBuilder result = new StringBuilder();
        if (hasValue) {
            //基本部分
            result.append(String.format("ALTER TABLE %s ADD COLUMN %s %s", table, name, type));
            //字段长度
            if (nonNull(length) && type.equals("java.lang.String")) {
                result.append(String.format("(%s)", length));
            }
            result.append(";");
            return result.toString();
        } else {
            log.error("表名或字段名不能为空");
            throw new RuntimeException("表名或字段名不能为空");
        }
    }

    /**
     * 修改表的单个字段的SQL语句
     *
     * @param table     表名
     * @param name      旧字段名
     * @param newName   新字段名
     * @param newType   字段类型
     * @param newLength 字段长度
     * @return SQL语句
     */
    public static String updateColumnType(String table, String name, String newName, String newType, Integer newLength, String driverType, Map<String, String> sqlTypeMap) {
        newType = sqlTypeMap.get(newType);
        boolean hasValue = !Strings.isNullOrEmpty(table)
                && nonNull(newType);
        StringBuilder result = new StringBuilder();
        if (hasValue) {
            if (needLengthColumn(newType)) {
                result.append(String.format("ALTER TABLE %s MODIFY %s %s (%s) ;", table, name, newType, newLength));
            } else {
                result.append(String.format("ALTER TABLE %s MODIFY %s %s ;", table, name, newType));
            }
            if (isNull(newName)) {
                result.append(String.format("ALTER TABLE %s RENAME COLUMN %s TO %s ;", table, name, newName));
            }
        } else {
            log.error("参数缺失");
        }
        return result.toString();
    }

    /**
     * 修改表的单个字段的SQL语句
     *
     * @param table   表名
     * @param name    旧字段名
     * @param newName 新字段名
     * @return SQL语句
     */
    public static String updateColumnName(String table, String name, String newName) {
        boolean hasValue = !Strings.isNullOrEmpty(table);
        StringBuilder result = new StringBuilder();
        if (hasValue) {
            if (!isNull(newName)) {
                result.append(String.format("ALTER TABLE %s RENAME COLUMN %s TO %s ;", table, name, newName));
            }
        } else {
            log.error("参数缺失");
        }
        return result.toString();
    }

    /**
     * 删除单个字段的SQL语句
     *
     * @param table 表名
     * @param name  字段名
     * @return SQL语句
     */
    public static String deleteColumn(String table, String name) {
        return String.format("ALTER TABLE %s DROP %s;", table, name);
    }

    private static boolean needLengthColumn(String type) {
        switch (type) {
            case "varchar":
                return true;
            default:
                return false;
        }
    }

}
