package com.tykj.classified.config.database;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateProperties;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateSettings;
import org.springframework.boot.autoconfigure.orm.jpa.JpaProperties;
import org.springframework.boot.orm.jpa.EntityManagerFactoryBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import javax.sql.DataSource;
import java.util.HashMap;
import java.util.Map;

@Configuration
@EnableTransactionManagement
@EnableJpaRepositories(
        entityManagerFactoryRef = "entityManagerFactoryOut",//配置连接工厂 entityManagerFactory
        transactionManagerRef = "transactionManagerOut", //配置 事物管理器  transactionManager
        basePackages = {"com.tykj.classified.dao.db2"}//设置持久层所在位置
)
public class OutDataSourceConfig {

    private final JpaProperties jpaProperties;

    private final HibernateProperties hibernateProperties;

    //自动注入配置好的数据源
    private final DataSource outDataSource;

    // 获取对应的数据库方言
    @Value("${spring.jpa.properties.hibernate.out-dialect}")
    private String outDialect;

    public OutDataSourceConfig(JpaProperties jpaProperties, HibernateProperties hibernateProperties, @Qualifier("dataSourceOut") DataSource outDataSource) {
        this.jpaProperties = jpaProperties;
        this.hibernateProperties = hibernateProperties;
        this.outDataSource = outDataSource;
    }

    /**
     * @param builder 构造器
     * @return bean
     */
    @Bean(name = "entityManagerFactoryOut")
    public LocalContainerEntityManagerFactoryBean entityManagerFactoryOut(EntityManagerFactoryBuilder builder) {
        Map<String, String> map = new HashMap<>();
        // 设置对应的数据库方言
        map.put("hibernate.dialect", outDialect);
        jpaProperties.setProperties(map);
        Map<String, Object> properties = hibernateProperties.determineHibernateProperties(
                jpaProperties.getProperties(), new HibernateSettings());
        return builder
                //设置数据源
                .dataSource(outDataSource)
                //设置数据源属性
                .properties(properties)
                //设置实体类所在位置.扫描所有带有 @Entity 注解的类
                .packages("com.tykj.classified.entity.db2")
                // Spring会将EntityManagerFactory注入到Repository之中.有了 EntityManagerFactory之后,
                // Repository就能用它来创建 EntityManager 了,然后 EntityManager 就可以针对数据库执行操作
                .persistenceUnit("outPersistenceUnit")
                .build();

    }

    /**
     * 配置事物管理器
     *
     * @param builder 构造器
     * @return manager
     */
    @Bean(name = "transactionManagerOut")
    PlatformTransactionManager transactionManagerOut(EntityManagerFactoryBuilder builder) {
        return new JpaTransactionManager(entityManagerFactoryOut(builder).getObject());
    }
}
