package com.tykj.classified.controller;

import com.tykj.classified.config.exception.BizException;
import com.tykj.classified.dao.db1.ConfClassifiedBasisDao;
import com.tykj.classified.dao.db1.ConfManageDetailDao;
import com.tykj.classified.dao.db1.EmployeeDao;
import com.tykj.classified.entity.db1.ConfClassifiedBasis;
import com.tykj.classified.entity.db1.ConfManageDetail;
import com.tykj.classified.entity.db1.Employee;
import com.tykj.classified.entity.vo.ConfClassifiedChangeVo;
import com.tykj.classified.response.ServerResponse;
import com.tykj.classified.service.ConfClassifiedChangeService;
import com.tykj.classified.utils.freemake.FreeMakerUtils;
import com.tykj.classified.utils.idwork.IdWorker;
import com.tykj.classified.utils.map.MapUtils;
import com.tykj.classified.utils.string.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author sjq
 */
@Slf4j
@RestController
@RequestMapping("/confClassifiedChange")
@Api(tags = "定密变更接口相关文档")
public class ConfClassifiedChangeController {

    private final ConfClassifiedChangeService confClassifiedChangeService;

    private final ConfManageDetailDao confManageDetailDao;

    private final EmployeeDao employeeDao;

    private final ConfClassifiedBasisDao confClassifiedBasisDao;

    private final IdWorker idWorker;

    @Value("${file.upload.path}")
    private String path;

    public ConfClassifiedChangeController(ConfClassifiedChangeService confClassifiedChangeService, ConfManageDetailDao confManageDetailDao, EmployeeDao employeeDao, ConfClassifiedBasisDao confClassifiedBasisDao, IdWorker idWorker) {
        this.confClassifiedChangeService = confClassifiedChangeService;
        this.confManageDetailDao = confManageDetailDao;
        this.employeeDao = employeeDao;
        this.confClassifiedBasisDao = confClassifiedBasisDao;
        this.idWorker = idWorker;
    }

    /**
     * title：点击变更查询定密变更信息
     * action：如果根据定密id查询到这个定密有变更信息就返回这个变更信息，如果没有就new一个变更信息将原定密部分信息放入变更信息中再返回给前端
     * @param classifiedId 定密id
     * @return ConfClassifiedChangeVo 定密变更Vo
     */
    @ApiOperation("根据定密id查询定密变更信息")
    @GetMapping("/findClassifiedChangeByClassifiedId")
    public ServerResponse<ConfClassifiedChangeVo> findClassifiedChangeByClassifiedId(@RequestParam("classifiedId") String classifiedId) {
        log.info("根据定密id查询定密变更信息===================================================");
        log.info("classifiedId:{}",classifiedId);
        ConfClassifiedChangeVo confClassifiedChangeVo = confClassifiedChangeService.findClassifiedChangeByClassifiedId(classifiedId);
        return ServerResponse.ok(confClassifiedChangeVo);
    }

    /**
     * title：定密变更保存按钮调用接口
     * action：新增一个定密变更信息记录，如果有定密变更id那就修改定密变更信息
     * @param confClassifiedChangeVo 定密变更Vo
     * @return String
     */
    @ApiOperation("保存定密变更信息")
    @PostMapping("/addClassifiedChange")
    public ServerResponse<String> addClassifiedChange(@RequestBody ConfClassifiedChangeVo confClassifiedChangeVo) {
        log.info("新增定密变更信息=======================================================");
        log.info("confClassifiedChangeVo:{}",confClassifiedChangeVo);
        confClassifiedChangeService.addClassifiedChange(confClassifiedChangeVo);
        return ServerResponse.ok("保存成功");
    }

    /**
     * title：定密变更审批通过按钮调用接口
     * action：定密变更审批通过做以下改变：
     * 1.定密信息的密级，保密期限，知悉范围都改为变更后的信息且定密到期时间延长，
     * 比如原定密到期时间是2022-08-30，变更后保密期限是1年，那么现定密到期时间在这个时间上延长一年
     * 2..删除这条定密变更记录
     * @param confClassifiedChangeVo 定密变更Vo
     * @return String
     */
    @ApiOperation("审批通过定密变更")
    @PostMapping("/approvalClassifiedChange")
    public ServerResponse<String> approvalClassifiedChange(@RequestBody ConfClassifiedChangeVo confClassifiedChangeVo) {
        log.info("定密变更审批通过=======================================================");
        log.info("confClassifiedChangeVo:{}",confClassifiedChangeVo);
        confClassifiedChangeService.approvalClassifiedChange(confClassifiedChangeVo);
        return ServerResponse.ok("变更审批通过");
    }

    /**
     * title：定密变更预览按钮调用接口
     * action：先将定密依据id，密级id，保密期限类型转化成对应名称再生成该定密变更的doc文档和pdf，并把pdf所在路径返回给前端
     * @param confClassifiedChangeVo 定密变更Vo
     * @return String
     */
    @ApiOperation("定密变更预览")
    @PostMapping("/classifiedChangePreview")
    public ServerResponse<String> classifiedChangePreview(@RequestBody ConfClassifiedChangeVo confClassifiedChangeVo) {
        log.info("定密变更预览============================================================");
        log.info("confClassifiedChangeVo:{}",confClassifiedChangeVo);
        // 定密依据
        List<ConfClassifiedBasis> confClassifiedBasisList = confClassifiedBasisDao.findAll();
        // 参数配置
        List<ConfManageDetail> confManageDetailList = confManageDetailDao.findAll();
        if (confClassifiedChangeVo.getBeforeClassification() != null) { // 变更前密级
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confManageDetail.getId().equals(confClassifiedChangeVo.getBeforeClassification()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedChangeVo.setBeforeClassification(collect.get(0).getManageOptions());
            }
        }
        if (confClassifiedChangeVo.getAfterClassification() != null) { // 变更后密级
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confManageDetail.getId().equals(confClassifiedChangeVo.getAfterClassification()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedChangeVo.setAfterClassification(collect.get(0).getManageOptions());
            }
        }
        if (confClassifiedChangeVo.getBeforeSecrecyTermType() == 1) { // 变更前保密期限类型：年
            confClassifiedChangeVo.setBeforeSecrecyTermContent(confClassifiedChangeVo.getBeforeSecrecyTermContent() + "年");
        } else if (confClassifiedChangeVo.getBeforeSecrecyTermType() == 2) { // 保密期限类型：月
            confClassifiedChangeVo.setBeforeSecrecyTermContent(confClassifiedChangeVo.getBeforeSecrecyTermContent() + "月");
        }
        if (confClassifiedChangeVo.getAfterSecrecyTermType() != null) {
            if (confClassifiedChangeVo.getAfterSecrecyTermType() == 1) { // 变更前保密期限类型：年
                confClassifiedChangeVo.setAfterSecrecyTermContent(confClassifiedChangeVo.getAfterSecrecyTermContent() + "年");
            } else if (confClassifiedChangeVo.getAfterSecrecyTermType() == 2) { // 保密期限类型：月
                confClassifiedChangeVo.setAfterSecrecyTermContent(confClassifiedChangeVo.getAfterSecrecyTermContent() + "月");
            }
        }
        if (confClassifiedChangeVo.getChangeBasisType() != null) { // 定密依据
            List<ConfClassifiedBasis> collect = confClassifiedBasisList.stream()
                    .filter(confClassifiedBasis -> confClassifiedBasis.getId().equals(confClassifiedChangeVo.getChangeBasisType()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedChangeVo.setChangeBasisType(collect.get(0).getBasisName());
            }
        }
        Map map = MapUtils.beanToMap(confClassifiedChangeVo);
        String word = new FreeMakerUtils().createWord(path,5, map, idWorker.nextId() + "");
        if (StringUtils.isEmpty(word)) {
            throw new BizException("生成文件失败");
        }
        return ServerResponse.ok(word);
    }
}
