package com.tykj.classified.controller;

import com.tykj.classified.config.exception.BizException;
import com.tykj.classified.dao.db1.ConfClassifiedBasisDao;
import com.tykj.classified.dao.db1.ConfManageDetailDao;
import com.tykj.classified.dao.db1.EmployeeDao;
import com.tykj.classified.entity.db1.ConfClassifiedBasis;
import com.tykj.classified.entity.db1.ConfManageDetail;
import com.tykj.classified.entity.db1.Employee;
import com.tykj.classified.entity.vo.*;
import com.tykj.classified.entity.vo.count.ClassifiedCountVo;
import com.tykj.classified.entity.vo.mythreadlocal.MyThreadLocal;
import com.tykj.classified.response.ServerResponse;
import com.tykj.classified.service.ConfClassifiedService;
import com.tykj.classified.utils.freemake.FreeMakerUtils;
import com.tykj.classified.utils.idwork.IdWorker;
import com.tykj.classified.utils.map.MapUtils;
import com.tykj.classified.utils.string.StringUtils;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author sjq
 */
@Slf4j
@RestController
@RequestMapping("/confClassified")
@Api(tags = "定密接口相关文档")
public class ConfClassifiedController {

    private final ConfClassifiedService confClassifiedService;

    private final ConfManageDetailDao confManageDetailDao;

    private final EmployeeDao employeeDao;

    private final ConfClassifiedBasisDao confClassifiedBasisDao;

    private final IdWorker idWorker;

    @Value("${file.upload.path}")
    private String path;

    public ConfClassifiedController(ConfClassifiedService confClassifiedService, ConfManageDetailDao confManageDetailDao, EmployeeDao employeeDao, ConfClassifiedBasisDao confClassifiedBasisDao, IdWorker idWorker) {
        this.confClassifiedService = confClassifiedService;
        this.confManageDetailDao = confManageDetailDao;
        this.employeeDao = employeeDao;
        this.confClassifiedBasisDao = confClassifiedBasisDao;
        this.idWorker = idWorker;
    }

    /**
     * title：所有定密信息列表的分页查询
     * action：查询列表信息，具体字段作用如下
     * 1.selectType：判断是确定列表还是变更解除列表还是所有事项列表，1-确定列表，2-变更或解除列表，null-所有事项列表
     * 确定列表只显示待定密的定密信息，变更或解除列表只显示保密中的定密信息，所有事项显示所有定密信息
     * 2.itemName，type，classification：头部模糊查询字段
     * 3.secretType：区分国密列表和工密列表。2-工作秘密，3-国密秘密
     * 4.specifiedTimeType：解密列表根据时间段查询。1-6月内到期，2-2年内到期，3-全部解密事项
     * 5.isType，isClassified，isStatus，classifiedTime，updateTime：按类型或密级或状态分组按顺序排序，顺序参照参数配置列表排序。也可以按classifiedTime保密到期时间排序
     * 刚进来默认按updateTime最近更新时间降序排序，
     * @param confClassifiedPageVo 定密分页查询Vo
     * @return ConfClassifiedVo 定密Vo
     */
    @ApiOperation("查询定密信息-分页(事项列表分页查询)")
    @PostMapping("/findAllClassifiedPaging")
    public ServerResponse<ConfClassifiedVo> findAllClassifiedPaging(@RequestBody ConfClassifiedPageVo confClassifiedPageVo) {
        log.info("查询定密信息-分页(事项列表分页查询)==============================================");
        log.info("confClassifiedPageVo:{}",confClassifiedPageVo);
        return ServerResponse.ok(confClassifiedService.findAllClassifiedPaging(confClassifiedPageVo));
    }

    /**
     * title：查询人员信息以及其他下拉列表信息
     * action：因为存的都是id，像密级，依据或者意见人都是存的id，所以查询详情信息的时候要把下拉列表的id和名称查询出来返给前端，在查询详情接口前调用此接口
     * @return EmployeeAndOtherOfGmVo 人员以及其他下拉框Vo
     */
    @ApiOperation(value = "查询人员信息以及其他下拉信息")
    @PostMapping("/findEmployeeAndOtherInfoOfGM")
    public ServerResponse<EmployeeAndOtherOfGmVo> findEmployeeAndOtherInfoOfGM(@RequestBody DownListVo downListVo) {
        log.info("查询人员信息以及其他下拉信息===============================================");
        return ServerResponse.ok(confClassifiedService.findEmployeeAndOtherInfoOfGM(downListVo));
    }

    @ApiOperation(value = "查询意见以及知悉范围下拉信息")
    @GetMapping("/findOption")
    public ServerResponse<ConfClassifiedOptionsVo> findOption() {
        log.info("查询人员信息以及其他下拉信息===============================================");
        ConfClassifiedOptionsVo confClassifiedOptionsVo = confClassifiedService.findOption();
        return ServerResponse.ok(confClassifiedOptionsVo);
    }

    /**
     * title：国密和工密保存定密信息
     * action：按照secretType区分保存的是国密还是工密，将字段对应的存入定密信息中保存（分主表和子表）
     * @param confClassifiedAndDetailsVo 定密主表与子表Vo
     * @param secretType 秘密类型
     * @return String
     */
    @ApiOperation("保存定密信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "secretType", value = "秘密类型(1-公开、2-工作秘密、3-国家秘密)",dataType = "int")
    })
    @PostMapping("/addClassified")
    public ServerResponse<String> addClassified(@RequestBody ConfClassifiedAndDetailsVo confClassifiedAndDetailsVo,@RequestParam("secretType") Integer secretType) {
        log.info("新增/修改定密信息==========================================================");
        log.info("conClassifiedAndDetailsVo,secretType:{},{}", confClassifiedAndDetailsVo,secretType);
        confClassifiedService.addClassified(confClassifiedAndDetailsVo,secretType);
        return ServerResponse.ok("保存成功");
    }

    /**
     * title：根据定密id查询定密详情
     * action：根据定密id查询定密详情并把信息存入ConfClassifiedAndDetailsVo中，其中子表的信息在content字段中用json存的，查询出来需要转到ConfClassifiedDetailsVo中
     * @param classifiedId 定密id
     * @return ConfClassifiedAndDetailsVo 定密主表与子表Vo
     */
    @ApiOperation("根据定密id查询定密详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "classifiedId", value = "定密id",dataType = "String")
    })
    @GetMapping("/findClassifiedInfo")
    public ServerResponse<ConfClassifiedAndDetailsVo> findClassifiedInfo(@RequestParam("classifiedId") String classifiedId) {
        log.info("查看待编辑定密详情======================================================");
        log.info("classifiedId:{}",classifiedId);
        return ServerResponse.ok(confClassifiedService.findClassifiedInfo(classifiedId));
    }

    /**
     * title：国密和工密定密预览保存按钮调用接口
     * action：按照secretType以及tye区分是国密的分发文拟稿，sm采购，sm事项还是工密预览，先将对应id转成名称再生成doc文档以及pdf，然后将pdf所在路径返回给前端
     * @param confClassifiedAndDetailsVo 定密主表与子表Vo
     * @return String
     */
    @ApiOperation("定密预览")
    @PostMapping("/classifiedPreview")
    public ServerResponse<String> classifiedPreview(@RequestBody ConfClassifiedAndDetailsVo confClassifiedAndDetailsVo) {
        // 承办人
        Employee employeeByPersonId = employeeDao.findEmployeeByPersonId(confClassifiedAndDetailsVo.getUndertaker());

        // 定密依据
        List<ConfClassifiedBasis> confClassifiedBasisList = confClassifiedBasisDao.findAll();
        // 参数配置
        List<ConfManageDetail> confManageDetailList = confManageDetailDao.findAll();
        if (confClassifiedAndDetailsVo.getClassifiedBasisType() == 1) { // 定密依据
            List<ConfClassifiedBasis> collect = confClassifiedBasisList.stream()
                    .filter(confClassifiedBasis -> confClassifiedBasis.getId().equals(confClassifiedAndDetailsVo.getClassifiedBasisContent()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedAndDetailsVo.setClassifiedBasisContent(collect.get(0).getCatalogueName() + "第" + collect.get(0).getSerialNo() + "条");
            }
        }
        if (confClassifiedAndDetailsVo.getClassification() != null) { // 密级
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confManageDetail.getId().equals(confClassifiedAndDetailsVo.getClassification()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedAndDetailsVo.setClassification(collect.get(0).getManageOptions());
            }
        }
        if (confClassifiedAndDetailsVo.getSecrecyTermType() == 1) { // 保密期限类型：年
            confClassifiedAndDetailsVo.setSecrecyTermContent(confClassifiedAndDetailsVo.getSecrecyTermContent() + "年");
        } else if (confClassifiedAndDetailsVo.getSecrecyTermType() == 2) { // 保密期限类型：月
            confClassifiedAndDetailsVo.setSecrecyTermContent(confClassifiedAndDetailsVo.getSecrecyTermContent() + "月");
        } else if (confClassifiedAndDetailsVo.getSecrecyTermType() == 3) { // 保密期限类型：日
            confClassifiedAndDetailsVo.setSecrecyTermContent(confClassifiedAndDetailsVo.getSecrecyTermContent() + "日");
        }
        if (confClassifiedAndDetailsVo.getUrgency() != null) { // 缓急程度
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confManageDetail.getId().equals(confClassifiedAndDetailsVo.getUrgency()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confClassifiedAndDetailsVo.setUrgency(collect.get(0).getManageOptions());
            }
        }
        Map map = MapUtils.beanToMap(confClassifiedAndDetailsVo);
        if (employeeByPersonId != null) {
            map.put("undertakerDept",employeeByPersonId.getUnitDepartment());
        }

        if (confClassifiedAndDetailsVo.getSecretType() == 3) { // gjmm
            if (confClassifiedAndDetailsVo.getType() == 1) {
                if (StringUtils.isEmpty(confClassifiedAndDetailsVo.getUndertakerUnitTwo()) &&
                        StringUtils.isEmpty(confClassifiedAndDetailsVo.getApprover())) {
                    String word = new FreeMakerUtils().createWord(path,1, map, idWorker.nextId() + "");
                    if (StringUtils.isEmpty(word)) {
                        throw new BizException("生成文件失败");
                    }
                    return ServerResponse.ok(word);
                } else if (!StringUtils.isEmpty(confClassifiedAndDetailsVo.getUndertakerUnitTwo()) &&
                        StringUtils.isEmpty(confClassifiedAndDetailsVo.getApprover())) {
                    String word = new FreeMakerUtils().createWord(path,8, map, idWorker.nextId() + "");
                    if (StringUtils.isEmpty(word)) {
                        throw new BizException("生成文件失败");
                    }
                    return ServerResponse.ok(word);
                } else if (StringUtils.isEmpty(confClassifiedAndDetailsVo.getUndertakerUnitTwo()) &&
                        !StringUtils.isEmpty(confClassifiedAndDetailsVo.getApprover())) {
                    String word = new FreeMakerUtils().createWord(path,9, map, idWorker.nextId() + "");
                    if (StringUtils.isEmpty(word)) {
                        throw new BizException("生成文件失败");
                    }
                    return ServerResponse.ok(word);
                } else {
                    String word = new FreeMakerUtils().createWord(path,10, map, idWorker.nextId() + "");
                    if (StringUtils.isEmpty(word)) {
                        throw new BizException("生成文件失败");
                    }
                    return ServerResponse.ok(word);
                }
            } else if (confClassifiedAndDetailsVo.getType() == 2) {
                String word = new FreeMakerUtils().createWord(path,2, map, idWorker.nextId() + "");
                if (StringUtils.isEmpty(word)) {
                    throw new BizException("生成文件失败");
                }
                return ServerResponse.ok(word);
            } else {
                String word = new FreeMakerUtils().createWord(path,3, map, idWorker.nextId() + "");
                if (StringUtils.isEmpty(word)) {
                    throw new BizException("生成文件失败");
                }
                return ServerResponse.ok(word);
            }
        } else if (confClassifiedAndDetailsVo.getSecretType() == 2) { // gzmm
            String word = new FreeMakerUtils().createWord(path,6, map, idWorker.nextId() + "");
            if (StringUtils.isEmpty(word)) {
                throw new BizException("生成文件失败");
            }
            return ServerResponse.ok(word);
        } else {
            throw new BizException("预览失败");
        }
    }

    /**
     * title：定密采购备案预览按钮调用接口
     * action：只有sm采购这个类型才有定密采购备案预览，先将id转化为名称再生成doc文档和pdf，把pdf所在路径返回给前端
     * @param confBaPreviewVo 定密备案预览Vo
     * @return String
     */
    @ApiOperation("定密采购备案预览")
    @PostMapping("/classifiedBaPreview")
    public ServerResponse<String> classifiedBaPreview(@RequestBody ConfBaPreviewVo confBaPreviewVo) {
        log.info("定密采购备案预览============================================================");
        log.info("confBaPreviewVo:{}",confBaPreviewVo);
        // 参数配置
        List<ConfManageDetail> confManageDetailList = confManageDetailDao.findAll();
        if (confBaPreviewVo.getPurchasingUnit() != null) { // 采购单位
            List<ConfManageDetail> collect = confManageDetailList.stream()
                    .filter(confManageDetail -> confManageDetail.getId().equals(confBaPreviewVo.getPurchasingUnit()))
                    .collect(Collectors.toList());
            if (collect.size() > 0) {
                confBaPreviewVo.setPurchasingUnit(collect.get(0).getManageOptions());
            }
        }
        Map map = MapUtils.beanToMap(confBaPreviewVo);
        String word = new FreeMakerUtils().createWord(path,4, map, idWorker.nextId() + "");
        if (StringUtils.isEmpty(word)) {
            throw new BizException("生成文件失败");
        }
        return ServerResponse.ok(word);
    }

    /**
     * title：定密审批通过按钮调用接口
     * action：审批通过做以下改变
     * 1.如果签发日期是当天或者更早就修改定密信息的changeStatus状态为2-保密中，再计算定密到期时间，
     * 根据签发日期作为开始保密时间，保密期限作为保密时长，如果保密期限是自定义的不能计算的日期那么定密到期时间为空不管。
     * 如果签发日期是当天之后，则通过定时器方法每天早上4点去执行（定时器没有暴露接口方法在service层）
     * @param confClassifiedAndDetailsVo 定密主表与子表Vo
     * @return String
     */
    @ApiOperation("审批通过定密")
    @PostMapping("/classifiedApproved")
    public ServerResponse<String> classifiedApproved(@RequestBody ConfClassifiedAndDetailsVo confClassifiedAndDetailsVo) {
        log.info("定密审批通过=========================================================");
        log.info("confClassifiedAndDetailsVo:{}",confClassifiedAndDetailsVo);
        confClassifiedService.classifiedApproved(confClassifiedAndDetailsVo);
        return ServerResponse.ok("审批通过");
    }

    /**
     * title：查询定密信息详情（基本信息，解密信息，操作记录）
     * action：根据定密id查询定密信息
     * @param classifiedId 定密id
     * @return ConfClassifiedInfoVo 定密所有信息Vo
     */
    @ApiOperation("查询定密基本信息、解密信息和操作记录(查看详情)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "classifiedId", value = "定密id",dataType = "String")
    })
    @GetMapping("/findClassifiedAllInfo")
    public ServerResponse<ConfClassifiedInfoVo> findClassifiedAllInfo(@RequestParam("classifiedId") String classifiedId) {
        log.info("查看定密所有信息详情========================================================");
        log.info("classifiedId:{}",classifiedId);
        return ServerResponse.ok(confClassifiedService.findClassifiedAllInfo(classifiedId));
    }

    /**
     * title:定密统计
     * action：统计定密信息。定密密级信息以及一年的新增/解密信息（按照签发日期和建议解密日期统计）
     * @return ClassifiedCountVo 定密统计Vo
     */
    @ApiOperation("统计定密")
    @GetMapping("/classifiedCount")
    public ServerResponse<ClassifiedCountVo> classifiedCount() {
        log.info("定密统计=====================================================================");
        return ServerResponse.ok(confClassifiedService.classifiedCount());
    }

    /**
     * title：初始化配置
     * */
    @ApiOperation("初始化配置")
    @GetMapping("/initManage")
    public ServerResponse<String> initManage() {
        log.info("初始化配置=====================================================================");
        confClassifiedService.initManage();
        return ServerResponse.ok("初始化配置成功");
    }
}
